/*
 * Arduino Controlled Servo Robot (SERB) - Test Program
 * For more details visit: http://www.oomlout.com/serb 
 * 
 * Behaviour: A simple test program which causes the SERB
 *            to turn randomly either left or right for a 
 *            random time period between 0.1 and 1 second.
 *            The SERB will then drive forward for a random 
 *            time period between 1 and 2 seconds. Finally
 *            pausing for 2 seconds before starting again.
 *
 * Wiring: Right Servo Signal - pin 9
 *         Left Servo Signal - pin 10 
 *
 * License: This work is licenced under the Creative Commons 
 *          Attribution-Share Alike 3.0 Unported License. To 
 *          view a copy of this licence, visit 
 *          http://creativecommons.org/licenses/by-sa/3.0/ 
 *          or send a letter to Creative Commons, 171 Second 
 *          Street, Suite 300, San Francisco, California 94105, 
 *          USA.
 *         
*/ 
 
//---------------------------------------------------------------------------------
// START OF ARDUINO CONTROLLED SERVO ROBOT (SERB) PREAMBLE
#include <Servo.h>

#define LEFTSERVOPIN  10
#define RIGHTSERVOPIN  9

#include "WProgram.h"
void setup();
void loop();
void turnRandom(int minTime, int maxTime);
void goForwardRandom(int minTime, int maxTime);
void serbSetup();
void setSpeed(int newSpeed);
void goForward();
void goBackward();
void goRight();
void goLeft();
void goStop();
Servo leftServo; 
Servo rightServo; 

int speed = 100; //sets the speed of the robot (both servos) a percentage between 0 and 100

// END OF ARDUINO CONTROLLED SERVO ROBOT (SERB) PREAMBLE
//---------------------------------------------------------------------------------


/*
 * sets pins to appropriate states and attaches servos. Then pauses
 * for 1 second before the program starts
*/ 
void setup()                  
{
  serbSetup();                       //sets the state of all neccesary pins and adds servos to your sketch
  randomSeed(analogRead(0));         //sets the random number seed with something mildly random
  delay(1000);
}

/*
 * turns the robot either left or right (randomly) for a period between 
 * 0.1 and 1 second. Before going forward for a random time period 
 * between 1 and 4 seconds. Before pausing for two seconds then starting
 * again.
*/
void loop()                     
{
   turnRandom(100,1000);            //Turns randomly left or right for a random time period between .1 second and one second
   goForwardRandom(1000,2000);      //Goes forward for a random time period between 1 and 2 seconds
   goStop();                        //Stops the robot
   delay(2000);                     //pauses for 2 seconds (whilst stopped)
}

/*
 * turns the robot randomly left or right for a random time period between
 * minTime (milliseconds) and maxTime (milliseconds)
 */
void turnRandom(int minTime, int maxTime){
  int choice = random(2);                     //Random number to decide between left (1) and right (0)
  int turnTime = random(minTime,maxTime);     //Random number for the pause time
  if(choice == 1){ goLeft();}                 //If random number = 1 then turn left
  else {goRight();}                           //If random number = 0 then turn right
  delay(turnTime);                            //delay for random time                         
}

/*
 * goes forward for a random time period between minTime (milliseconds)
 * and maxTime (milliseconds)
 */
void goForwardRandom(int minTime, int maxTime){
  int forwardTime = random(minTime,maxTime);      //determine a random time to go forward
  goForward();                                    //sets the SERB forward
  delay(forwardTime);                             //delays for random time period
}

//------------------------------------------------------------------------------------------------------------
//START OF ARDUINO CONTROLLED SERVO ROBOT (SERB) ROUTINES

/*
 * sets up your arduino to address your SERB using the included routines
*/
void serbSetup(){
  setSpeed(speed);
  pinMode(LEFTSERVOPIN, OUTPUT);     //sets the left servo signal pin to output
  pinMode(RIGHTSERVOPIN, OUTPUT);    //sets the right servo signal pin to output
  leftServo.attach(LEFTSERVOPIN);    //attaches left servo
  rightServo.attach(RIGHTSERVOPIN);  //attaches right servo
  goStop();
}


/*
 * sets the speed of the robot between 0-(stopped) and 100-(full speed)
 * NOTE: speed will not change the current speed you must change speed 
 * then call one of the go methods before changes occur.
*/ 
void setSpeed(int newSpeed){
  if(newSpeed >= 100) {newSpeed = 100;}     //if speed is greater than 100 make it 100
  if(newSpeed <= 0) {newSpeed = 0;}         //if speed is less than 0 make it 0 
  speed = newSpeed * 0.9;                   //scales the speed to be between 0 and 90
}

/*
 * sends the robot forwards
 */
void goForward(){
 leftServo.write(90 + speed);
 rightServo.write(90 - speed);
}

/*
 * sends the robot backwards
 */
void goBackward(){
 leftServo.write(90 - speed);
 rightServo.write(90 + speed);
}
  
/*
 * sends the robot right
 */
void goRight(){
 leftServo.write(90 + speed);
 rightServo.write(90 + speed);
}

/*
 * sends the robot left
 */
void goLeft(){
 leftServo.write(90 - speed);
 rightServo.write(90 - speed);
}

/*
 * stops the robot
 */
void goStop(){
 leftServo.write(90);
 rightServo.write(90);
} 
//END OF ARDUINO CONTROLLED SERVO ROBOT (SERB) ROUTINES
//------------------------------------------------------------------------------------------------------------

int main(void)
{
	init();

	setup();
    
	for (;;)
		loop();
        
	return 0;
}

