/*
 ESP-NOW based sensor station

 Sends readings to an ESP-Now server with a fixed mac address

 Original author:  Anthony Elder
 License: Apache License v2

 December 2018 Modified by MarcellS2
 
 Uses Sparkfun library for BME280
 For Lolin ESP8266 Board or similar ESP8266 modules
 Connect the pins in the following way:
 *
 * Generic ESP-12E        GY-BME280 breakout board
 * ==================================
 *   (D1) GPIO5            SCL (pin 3)
 *   (D2) GPIO4            SDA (pin 4)
 *   3.3V                  VDD (pin 1)
 *   GND                   GND (pin 2)
 *
 * ===================================
 * 
 * Also measures the light level uses an analog sensor (CdS based)
 * Input is 0 (brightest) to 1024 (darkest) -> will invert for user and set as a percentage for now.
 * 
*/

#include <ESP8266WiFi.h>
#include <Wire.h>
#include "SparkFunBME280.h"

extern "C" {
  #include <espnow.h>
//  #include "user_interface.h"
}

// this is the MAC Address of the remote ESP server which receives these sensor readings
// it is an internal network (private) MAC address
uint8_t remoteMac[] = {0x36, 0x33, 0x33, 0x33, 0x33, 0x33};

#define WIFI_CHANNEL 1
//#define SLEEP_SECS 15 * 60 // 15 minutes
#define SLEEP_SECS 300  // sleep in seconds
#define SEND_TIMEOUT 100  // 245 millis seconds timeout 

#define BME280_ADDRESS (0x76) // GY-BME280 board has this address
#define SEALEVELPRESSURE_HPA (1013.25)


// keep in sync with slave struct
struct __attribute__((packed)) SENSOR_DATA {
  float temp;
  float humidity;
  float pressure;
  float lightlvl;
} sensorData;

//Create an instance of the object
BME280 bme280;  // I2C

void setup() {
  Serial.begin(115200); Serial.println();
  
  //Wire.pins(int sda, int scl)
   Wire.pins(4, 5); //on Generic ESP-12E.

  Serial.println(" -- Serial Init done --");

  // Read sensor measurements and fill in mock data for items not measured by SI7021 
  // Other sensors may be added to complete this in the future.
  readSensor();

  if (esp_now_init() != 0) {
    Serial.println("*** ESP_Now init failed");
    gotoSleep();
  }
  Serial.println(" -- ESP_Now init success --");

  WiFi.setOutputPower(20.5);

  esp_now_set_self_role(ESP_NOW_ROLE_CONTROLLER);
  esp_now_add_peer(remoteMac, ESP_NOW_ROLE_SLAVE, WIFI_CHANNEL, NULL, 0);

  esp_now_register_send_cb([](uint8_t* mac, uint8_t sendStatus) {
    Serial.printf("send_cb, send done, status = %i\n", sendStatus);
    gotoSleep();
  });

  // send the data to the ESP-Now Hub
  uint8_t bs[sizeof(sensorData)];
  memcpy(bs, &sensorData, sizeof(sensorData));
  esp_now_send(remoteMac, bs, sizeof(sensorData)); // NULL means send to all peers
  Serial.println(" -- ESP_Now data sent --");
}

void loop() {
  if (millis() > SEND_TIMEOUT) {
    gotoSleep();
  }
}

void readSensor() {

  // set default values so faulty sensor is noticeable
  sensorData.temp = -100;
  sensorData.humidity = -100;
  sensorData.pressure = -100;
  sensorData.lightlvl = -100;
  readBME280();
  measurelight();
  Serial.printf("temp=%01f, humidity=%01f, pressure=%01f, lightlvl=%01f\n", 
       sensorData.temp, sensorData.humidity, sensorData.pressure, sensorData.lightlvl);
}

void readBME280() {
  bme280.settings.commInterface = I2C_MODE;
  bme280.settings.I2CAddress = 0x76;
  bme280.settings.runMode = 2; // Forced mode with deepSleep
  bme280.settings.tempOverSample = 1;
  bme280.settings.pressOverSample = 1;
  bme280.settings.humidOverSample = 1;
  Serial.print("bme280 init="); Serial.println(bme280.begin(), HEX);
  sensorData.temp = bme280.readTempC();
  sensorData.humidity = bme280.readFloatHumidity();
  sensorData.pressure = bme280.readFloatPressure() / 100.0;
  Serial.printf("temp=%01f, humidity=%01f, pressure=%01f\n", sensorData.temp, sensorData.humidity, sensorData.pressure);
}

void gotoSleep() {
  // add some randomness to avoid collisions with multiple devices
  int sleepSecs = SLEEP_SECS + ((uint8_t)RANDOM_REG32/8); 
//  int sleepSecs = SLEEP_SECS; 
  Serial.printf("Awake for %i ms, going to sleep for %i secs...\n", millis(), sleepSecs);
  // deep sleep time is given in microseconds
  ESP.deepSleep(sleepSecs * 1000000, RF_NO_CAL);
}

void measurelight() {
  int adc_val;

  adc_val = analogRead(A0);
  sensorData.lightlvl = (float) (1024-adc_val)/1024.0 * 100.0;
  
}
