/*
 ESP-NOW based sensor 

 Sends readings to an ESP-Now server with a fixed mac address

 Original Author:  Anthony Elder
 License: Apache License v2
 
 December 2018
 Adapted for use with ESP-01 boards

 This board will use the SI7021 (or similar sensor module)
 
  ESP8266-01          SI7021
 * ==================================
 *   GPIO2              SCL (pin 6)
 *   GPIO0              SDA (pin 1)
 *   3.3V               VDD (pin 5)
 *   GND                GND (pin 2)
 * ====================================
 *
*/

#include <ESP8266WiFi.h>
#include <Wire.h>
#include "SI7021.h"

extern "C" {
  #include <espnow.h>
//  #include "user_interface.h"
}

// this is the MAC Address of the remote ESP server which receives these sensor readings
// it is an internal network (private) MAC address
uint8_t remoteMac[] = {0x36, 0x33, 0x33, 0x33, 0x33, 0x33};

#define WIFI_CHANNEL 1
//#define SLEEP_SECS 15 * 60 // 15 minutes
#define SLEEP_SECS 300  // sleep in seconds
#define SEND_TIMEOUT 100  // 245 millis seconds timeout 

// keep in sync with slave struct
struct __attribute__((packed)) SENSOR_DATA {
  float temp;
  float humidity;
  float pressure;
  float lightlvl;
} sensorData;

//Create an instance of the object
SI7021 myHumidity;

void setup() {
  Serial.begin(115200); Serial.println();
  
  //Wire.pins(int sda, int scl)
  //Wire.pins(0, 2); //on ESP8266-01.
    myHumidity.begin(0, 2);

  Serial.println(" -- Serial Init done. Set up mock temp data --");

  // Read sensor measurements and fill in mock data for items not measured by SI7021 
  // Other sensors may be added to complete this in the future.
  readSensor();

  if (esp_now_init() != 0) {
    Serial.println("*** ESP_Now init failed");
    gotoSleep();
  }
  Serial.println(" -- ESP_Now init success --");

  esp_now_set_self_role(ESP_NOW_ROLE_CONTROLLER);
  esp_now_add_peer(remoteMac, ESP_NOW_ROLE_SLAVE, WIFI_CHANNEL, NULL, 0);

  esp_now_register_send_cb([](uint8_t* mac, uint8_t sendStatus) {
    Serial.printf("send_cb, send done, status = %i\n", sendStatus);
    gotoSleep();
  });

  // send the data to the ESP-Now Hub
  uint8_t bs[sizeof(sensorData)];
  memcpy(bs, &sensorData, sizeof(sensorData));
  esp_now_send(remoteMac, bs, sizeof(sensorData)); // NULL means send to all peers
  Serial.println(" -- ESP_Now data sent --");
}

void loop() {
  if (millis() > SEND_TIMEOUT) {
    gotoSleep();
  }
}

void readSensor() {
  sensorData.temp = myHumidity.readTemperature();
  sensorData.humidity = myHumidity.readHumidity();
  sensorData.pressure = -100;
  sensorData.lightlvl = -100;
  Serial.printf("temp=%01f, humidity=%01f, pressure=%01f, lightlvl=%01f\n", 
       sensorData.temp, sensorData.humidity, sensorData.pressure, sensorData.lightlvl);
}

void gotoSleep() {
  // add some randomness to avoid collisions with multiple devices
  int sleepSecs = SLEEP_SECS + ((uint8_t)RANDOM_REG32/8); 
//  int sleepSecs = SLEEP_SECS; 
  Serial.printf("Awake for %i ms, going to sleep for %i secs...\n", millis(), sleepSecs);
  // deep sleep time is given in microseconds
  ESP.deepSleep(sleepSecs * 1000000, RF_NO_CAL);
}
