/* Touch Screen Geiger Counter Sketch
   v1.0 by K Baker December 2013
   
   This sketch is a highly modified version of GeigerKit_Default sketch (v.10.2) by bHogan.
   The program was simplifed to run specifically on my Touch Screen Geiger Counter and 
   code added for the capactive touch switches and LED effects.
   
   The Geiger hardware and base code was from here: https://sites.google.com/site/diygeigercounter/
   Please support it by buying the hardware (its really good!)
  
   This program is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 2.1 of the License, or any later version.
   This program is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
   PARTICULAR PURPOSE.  See the GNU General Public License for more details.
   Do not remove information from this header.
   
   THIS PROGRAM AND IT'S MEASUREMENTS IS NOT INTENDED TO GUIDE ACTIONS TO TAKE, OR NOT
   TO TAKE, REGARDING EXPOSURE TO RADIATION. THE GEIGER KIT AND IT'S SOFTWARE ARE FOR
   EDUCATIONAL PURPOSES ONLY. DO NOT RELY ON THEM IN HAZARDOUS SITUATIONS!
 */
 
///////////////////////////////////////////////////////////////////////////////////////////////
// SETUP PARAMETERS
///////////////////////////////////////////////////////////////////////////////////////////////

#include <Arduino.h>
#include <LedControl.h>
#include <CapacitiveSensor.h>

//                             DEBUG DEFINES
#define DEBUG          false            // if true, shows available memory and other debug info
#define SELF_TEST      false            // if true, adds one to each counter every 167ms - simulates a ~360CPM count

// PIN MAP - Each I/O pin (used or unused) is defined . . .

#define MUTE               A5              // Mute/unmute button
#define LED_BED            A4              // Banana Equivalent Dose LED
#define MAX7219_LOAD       A3              // MAX7219 pin 12
#define PIEZO_SIG_PIN      A2              // (A2) signal pin to control piezo
#define THRESHOLD_SIGNAL   A1              // the input signal for the threshold trimpot
#define THRESHOLD_SENSE    A0              // the output signal for the threshold trimpot

//                         D13
#define MAX7219_DIN        12              // MAX7219 pin 1 (Make sure CLICK/TONE jumper is set to CLICK!)
#define MAX7219_CLK        11              // MAX7219 pin 13
#define LED_THROBBER       10              // LED throbber
#define LED_CPM             9              // Counts per minute LED
#define LED_uSV             8              // uSv LED
#define SEL_uSV             7              // uSv units button 
#define SEL_CPM             6              // CPM units button
#define LED_MUTE            5              // Mute button LED
#define CAPCOMMON           4              // Common pin for switches
#define SEL_BED             3              // BED units button
//                         D2               Interrupt 0 for Geiger 
//                         D1 & D0          serial comm

// other defines . . .
#define LOW_VCC         4200 //mV    // if Vcc < LOW_VCC give low voltage warning
#define DISP_PERIOD     5000            // defaults to 5 sec sample & display
#define LOGGING_PERIOD  5000            // defaults to 5 sec logging period
#define DOSE_RATIO       227.44         // 175.43 for SBM-20, 227.44 for SBM-20 & SI-180G in parallel
#define BLINKPERIOD     2000            // defaults to 2 second blink period
#define FADEPERIOD      1500            // defaults to 1.5 seconds for the throbber to fade
#define UNIT_uSV           0            // units to display 
#define UNIT_CPM           1            // " "
#define UNIT_BED           2            // " "
#define TOGGLEDELAY      750            // default of 3/4 second for switch standdown period
#define SWPRESS          500            // how long a button must be pressed for effect

////////////////////////////////// globals /////////////////////////////////////

// low battery
boolean lowVcc = false;                 // true when Vcc < LOW_VCC

// variables for counting periods and counts . . .
unsigned long dispPeriodStart;          // for display period
volatile unsigned long dispCnt;         // to count and display CPM
unsigned long logPeriodStart;           // for logging period
volatile unsigned long logCnt;          // to count and log CPM
unsigned int units=UNIT_uSV;            // units to display in (uSV, CPM, BED)
unsigned long blinkPeriodStart;         // for blinking low voltage LEDs
volatile unsigned long faderPeriodStart;// for LED throbber

// digit display LEDs
LedControl lc=LedControl(MAX7219_DIN,MAX7219_CLK,MAX7219_LOAD,1);  // 2x4 7-segment display.  A5 - 7219 pin 1, A4 - pin 13, A3 - pin 12.  All acting as digital pins

// throbber
volatile int faderValue = 0;

// Capacitive switches
CapacitiveSensor cs_uSv = CapacitiveSensor(CAPCOMMON,SEL_uSV);
CapacitiveSensor cs_CPM = CapacitiveSensor(CAPCOMMON,SEL_CPM);
CapacitiveSensor cs_BED = CapacitiveSensor(CAPCOMMON,SEL_BED);
CapacitiveSensor cs_Mute = CapacitiveSensor(CAPCOMMON,MUTE);
long usvPressTime = 0;
long cpmPressTime = 0;
long bedPressTime = 0;
long mutePressTime = 0;
long toggleStart = millis();  // delay when switch toggled to allow time to settle
long piezoStart = toggleStart;
boolean PiezoOn = true;                 // preset to piezo = ON
int capThreshold = 0;

// display values
float dispUSv = 0.0;                      // display CPM converted to VERY APPROXIMATE uSv
float dispBed = 0.0;                      // display BED converted from uSv
unsigned long dispCPM;                    // display CPM

/////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////

void setup(){
  Serial.begin(9600);                   // comspec 96,N,8,1
  
  pinMode(LED_uSV, OUTPUT);
  pinMode(LED_CPM, OUTPUT);
  pinMode(LED_BED, OUTPUT);
  pinMode(LED_MUTE, OUTPUT);
  pinMode(LED_THROBBER, OUTPUT);
  
  attachInterrupt(0,GetEvent,FALLING);  // Geiger event on pin 2 triggers interrupt

  // LED digit display
  lc.shutdown(0,false);                 // turn off power saving, enables display
  lc.setIntensity(0,8);                 // sets brightness (0~15 possible values)
  lc.clearDisplay(0);                   // clear screen
  
  Serial.print(F("CPM,uSv/h,Vcc\r\n"));             // print header for CSV output to serial
  
  dispPeriodStart = millis();           // start timing display CPM
  logPeriodStart = dispPeriodStart;     // start logging timer
  blinkPeriodStart = dispPeriodStart;   // start blinking LED timer
  dispCnt = 0;                          // start with fresh totals
  logCnt= 0;
  delay(TOGGLEDELAY);
  toggleUnitLEDs(UNIT_uSV);
  togglePiezo(PiezoOn);
  
  // trimpot sensor for capacitive touch threshold
  pinMode(THRESHOLD_SIGNAL, OUTPUT);
  pinMode(THRESHOLD_SENSE, INPUT);
  // read trim pot and set capacitive touch threshold
  digitalWrite(THRESHOLD_SIGNAL,HIGH);
  capThreshold = analogRead(THRESHOLD_SENSE);
  #if (DEBUG) 
    Serial.print("Cap Threshold: ");
    Serial.println(capThreshold);
  #endif
  digitalWrite(THRESHOLD_SIGNAL,LOW);
}


void loop(){ 
  // when a unit button is selected it toggles off the other two
  long sensorUnits[3];  // holds the values from the units buttons 
  sensorUnits[0] = cs_uSv.capacitiveSensor(30);  // uSv button
  sensorUnits[1] = cs_CPM.capacitiveSensor(30);  // CPM button
  sensorUnits[2] = cs_BED.capacitiveSensor(30);  // BED button
  
  // mute button is independant of the unit buttons
  long sensorMute = cs_Mute.capacitiveSensor(30); // Mute button

  //5 lines below for self check - you should see very close to 360 CPM
#if (SELF_TEST)
  dispCnt++;
  logCnt++;
  delay(167);                           // 167 mS = 6 Hz `= X 60 = 360 CPM  
#endif

  // check for capacitive switch changes for unit buttons and toggle them
  // a button has to be held for SWPRESS milliseconds to take effect
  // this is to reduce false positives caused by circuit noise
  switch(getIndexOfMaximumValue(sensorUnits,3)) {  // detects which sensor button has max value (it is being pushed)
    
    case 0:
    if (sensorUnits[0] > capThreshold) {
      if (usvPressTime == 0) {
        usvPressTime = millis();
      } else if (millis() >= usvPressTime + SWPRESS) {
        toggleUnitLEDs(UNIT_uSV);
        usvPressTime = 0;
      }
      #if (DEBUG)
        Serial.print("usvPressed: ");
        Serial.println(usvPressTime);
      #endif 
    } else {
      usvPressTime = 0;
    }
    break;
    
    case 1:
    if (sensorUnits[1] > capThreshold) {
      if (cpmPressTime == 0) {
        cpmPressTime = millis(); 
      } else if (millis() >= cpmPressTime + SWPRESS) {
        toggleUnitLEDs(UNIT_CPM);
        cpmPressTime = 0;
      }
      #if (DEBUG)
        Serial.print("cpmPressed: ");
        Serial.println(cpmPressTime);
      #endif 
    } else {
      cpmPressTime = 0;
    }  
    break;
    
    case 2:
    if (sensorUnits[2] > capThreshold) {
      if (bedPressTime == 0) {
        bedPressTime = millis();
      } else if (millis() >= bedPressTime + SWPRESS) {
        toggleUnitLEDs(UNIT_BED);
        bedPressTime = 0;
      }
      #if (DEBUG)
        Serial.print("bedPressed: ");
        Serial.println(bedPressTime);
      #endif 
    } else {
      bedPressTime = 0;
    }
    break;
  }
  
  // similar code to check for the mute button.  Toggles independantly
  // from the others
  if (sensorMute > capThreshold) {
    if (mutePressTime == 0) {
      mutePressTime = millis();
    } else if (millis() >= mutePressTime + SWPRESS) {
      togglePiezo(!PiezoOn);
      mutePressTime = 0;  
    }
    #if (DEBUG)
      Serial.print("mutePressed: ");
      Serial.println(mutePressTime);
    #endif 
  } else {
    mutePressTime = 0;
  }

  // throbber code
  if (millis() >= faderPeriodStart + (FADEPERIOD / 255) && faderValue > 0) {  
    analogWrite(LED_THROBBER, faderValue);
    faderValue-=10;
    if (faderValue <= 0) faderValue = 0;
    faderPeriodStart = millis();
  }
  
  // calulates display and log values
  if (millis() >= dispPeriodStart + DISP_PERIOD){ // DISPLAY PERIOD
    if (readVcc() <= LOW_VCC) lowVcc = true; // check if Vcc is low
    else lowVcc = false;

    DispCounts(dispCnt);              // period is over - display counts
    dispCnt = 0;                        // reset counter
    dispPeriodStart = millis();         // reset the period time  
  }

  if (millis() >= logPeriodStart + LOGGING_PERIOD){ // LOGGING PERIOD
    logCount(logCnt);                   // pass in the counts to be logged
    logCnt = 0;                         // reset log event counter
    logPeriodStart = millis(); // reset log time and display time too
  }
  
  // blink LEDs if battery is low
  if (lowVcc && millis() >= blinkPeriodStart + BLINKPERIOD) {
    // blink switch LEDs about every two seconds
    Blinkleds();
    blinkPeriodStart = millis();
  }
}

void DispCounts(unsigned long dcnt){    // calc and display predicted CPM & uSv/h
  static float avgCnt;                  // holds the previous moving average count
  static byte sampleCnt;                // the number of samples making up the moving average
  byte maxSamples = (60000 / DISP_PERIOD) / 2;   // number of sample periods in 30 seconds   
  
  sampleCnt++;                                  // inc sample count - must be at least 1
  avgCnt += (dcnt - avgCnt) / sampleCnt;        // CALCULATE AVERAGE COUNT - moving average
  dispCPM = (avgCnt * 60000.0) / DISP_PERIOD;   // convert to CPM

  //handle reset of sample count - sample is for 1/2 min and reset. Options for reset value are:
  // "0" - throw away last average, "1" - keeps last average, "maxSamples -1" - keeps running avg.
  if (sampleCnt >= maxSamples) sampleCnt = 0;   // start a fresh average every 30 sec.

  // The following line gives a faster response when counts increase or decrease rapidly 
  // It resets the running average if the rate changes by  +/- 35% (previously it was 9 counts)
  if ((dcnt - avgCnt) > (avgCnt * .35) || (avgCnt - dcnt) > (avgCnt * .35)) sampleCnt = 0;
  dispUSv = float(dispCPM) / DOSE_RATIO;     // make dose rate conversion to uSv/hr
  dispBed = dispUSv / 0.1;                   // Banana Equivalent Dose is approx 0.1uSv
  
  if (units == UNIT_CPM) {
    lcdprint(dispCPM, false);
  } else if (units == UNIT_uSV) {
    lcdprint(dispUSv, true);
  } else if (units == UNIT_BED) {
    lcdprint(dispBed, true);
  }
}

void logCount(unsigned long lcnt){ // unlike logging sketch, just outputs to serial
  unsigned long logCPM;                 // log CPM
  float uSvLogged = 0.0;                // logging CPM converted to "unofficial" uSv

  logCPM = float(lcnt) / (float(LOGGING_PERIOD) / 60000);
  uSvLogged = (float)logCPM / DOSE_RATIO; // make uSV conversion

  // Print to serial in a format that might be used by Excel
  Serial.print(logCPM,DEC);
  Serial.write(',');    
  Serial.print(uSvLogged,4);
  Serial.write(','); // comma delimited
  Serial.print(readVcc()/1000. ,2);   // print as volts with 2 dec. places
  Serial.print(F("\r\n"));
}

unsigned long readVcc() { // SecretVoltmeter from TinkerIt
  unsigned long result;
  // Read 1.1V reference against AVcc
  ADMUX = _BV(REFS0) | _BV(MUX3) | _BV(MUX2) | _BV(MUX1);
  delay(2); // Wait for Vref to settle
  ADCSRA |= _BV(ADSC); // Convert
  while (bit_is_set(ADCSRA,ADSC));
  result = ADCL;
  result |= ADCH<<8;
  result = 1126400L / result; // Back-calculate AVcc in mV
//  lastVCC = result;           // TO DO - NOW MAKING ENOUGH CALLS TO WARRENT THIS - USING W/ METER
  return result;
}

///////////////////////////////// UTILITIES ///////////////////////////////////

int getIndexOfMaximumValue(long* array, int size){  // returns the index of the maximum value in an array
  int maxIndex = 0;
  int max = array[maxIndex];
  for (int i=1; i<size; i++){
    if (max<array[i]){
      max = array[i];
      maxIndex = i;
    }
  }
  return maxIndex;
}

void lcdprint(float num, boolean decplaces) {
/* Prints a floating point number on 8 digit 7-segment display.
   Automatically adjusts decimal point for size of number */
  char lcdigits[8];                 // the digits to be displayed on the lcd 
  String lcstring;
  unsigned int decimalplaces;
  unsigned int numlength;
  
  lc.clearDisplay(0);                          // clear the screen
  if (num > 99999999) {
    // the value is greater than the display can handle
    lc.setChar(0,4,'0',false); // Over Flow
    lc.setChar(0,3,'F',false);
  } else {
    if (num > 9999999 || decplaces == false) {                // show up to two decimal places if possible
      dtostrf(num,1,0,lcdigits);
      decimalplaces = 0;
    } else if (num > 999999) {
      dtostrf(num*10,1,0,lcdigits);
      decimalplaces = 1;
    } else {
      dtostrf(num*100,1,0,lcdigits);
      decimalplaces = 2;
    }
    lcstring=lcdigits;
    numlength=lcstring.length();
    // set the leds for each digit 
    for (int a=0; a < numlength; a++) {
      lc.setChar(0,a,lcdigits[numlength-a-1], (a == decimalplaces)); // print digit and decimal place
    }
    if (numlength <= decimalplaces) {  // if num < 0 add leading zeros to display
      for (int a=numlength; a <= decimalplaces; a++) {
        lc.setDigit(0,a,0,(a == decimalplaces));
      }
    }
  }
}

void Blinkleds(){ // flash the LEDs
  digitalWrite(LED_uSV, HIGH);
  digitalWrite(LED_CPM, HIGH);
  digitalWrite(LED_BED, HIGH);
  digitalWrite(LED_MUTE,HIGH);
  delay(150);
  if (units != UNIT_uSV) digitalWrite(LED_uSV, LOW);
  if (units != UNIT_CPM) digitalWrite(LED_CPM, LOW);
  if (units != UNIT_BED) digitalWrite(LED_BED, LOW);
  if (!PiezoOn) digitalWrite(LED_MUTE,LOW);
}

static void togglePiezo(boolean bState){      // toggle piezo control pin
  if (piezoStart + TOGGLEDELAY < millis()) {
    PiezoOn = bState;
    if (PiezoOn) {                        // if ON - set the pin to float 
      pinMode(PIEZO_SIG_PIN, INPUT);      // set the pin to input to make it float (high Z)
      digitalWrite(LED_MUTE, HIGH);       // turn on Mute button LED
    } 
    else {                              // it's OFF - set the pin to LOW
      pinMode(PIEZO_SIG_PIN, OUTPUT);
      digitalWrite(PIEZO_SIG_PIN,LOW);
      digitalWrite(LED_MUTE, LOW);       // turn off Mute button LED
    }
    piezoStart = millis();
  }
}

static void toggleUnitLEDs(unsigned int unitx) { // toggle units buttons
  if (toggleStart + TOGGLEDELAY < millis()) {
    units=unitx;
    if (units == UNIT_uSV) {
      digitalWrite(LED_uSV, HIGH);
      digitalWrite(LED_CPM, LOW);
      digitalWrite(LED_BED, LOW);
      lcdprint(dispUSv,true);
    } else if (units == UNIT_CPM) {
      digitalWrite(LED_uSV, LOW);
      digitalWrite(LED_CPM, HIGH);
      digitalWrite(LED_BED, LOW);  
      lcdprint(dispCPM,false);
    } else if (units == UNIT_BED) {
      digitalWrite(LED_uSV, LOW);
      digitalWrite(LED_CPM, LOW);
      digitalWrite(LED_BED, HIGH);
      lcdprint(dispBed,true);
    }      
    toggleStart = millis();
  }
}
  
///////////////////////////////// ISR ///////////////////////////////////
void GetEvent(){   // ISR triggered for each new event (count)
  dispCnt++;
  logCnt++;
  analogWrite(LED_THROBBER, 0);  // reset fader LED
  delay(10);
  faderValue = 255;
  analogWrite(LED_THROBBER, faderValue);
  faderPeriodStart = millis();
}




