// A simple data logger for the Adafruit Data Logging shield on a mega
// You must edit SdFatConfig.h and set MEGA_SOFT_SPI nonzero
#include <SdFat.h>
#include <SdFatUtil.h>  // define FreeRam()
#include <I2cMaster.h>
#include <SoftRTClib.h>
#include <OneWire.h>
#include <DallasTemperature.h>

#define CHIP_SELECT     10  // SD chip select pin
#define LOG_INTERVAL  300000  // mills between entries
#define ECHO_TO_SERIAL   0  // echo data to serial port if nonzero
#define WAIT_TO_START    0  // Wait for serial input in setup()
#define ADC_DELAY       10  // ADC delay for high impedence sensors

#if defined(__AVR_ATmega1280__) || defined(__AVR_ATmega2560__)
#if !MEGA_SOFT_SPI
#error set MEGA_SOFT_SPI nonzero in libraries/SdFat/SdFatConfig.h
#endif  // MEGA_SOFT_SPI
// Is a Mega use analog pins 4, 5 for software I2C
const uint8_t RTC_SCL_PIN = 59;
const uint8_t RTC_SDA_PIN = 58;
SoftI2cMaster i2c(RTC_SDA_PIN, RTC_SCL_PIN);

#elif defined(__AVR_ATmega32U4__)
#if !LEONARDO_SOFT_SPI
#error set LEONARDO_SOFT_SPI nonzero in libraries/SdFat/SdFatConfig.h
#endif  // LEONARDO_SOFT_SPI
// Is a Leonardo use analog pins 4, 5 for software I2C
const uint8_t RTC_SCL_PIN = 23;
const uint8_t RTC_SDA_PIN = 22;
SoftI2cMaster i2c(RTC_SDA_PIN, RTC_SCL_PIN);

#else  // defined(__AVR_ATmega1280__) || defined(__AVR_ATmega2560__)
// Not Mega use hardware I2C
// enable pull-ups on SDA/SCL
TwiMaster i2c(true);
#endif  // defined(__AVR_ATmega1280__) || defined(__AVR_ATmega2560__)

RTC_DS1307 RTC(&i2c); // define the Real Time Clock object
// file system object
SdFat sd;

// text file for logging
ofstream logfile;

// Serial print stream
ArduinoOutStream cout(Serial);

// buffer to format data - makes it eaiser to echo to Serial
char buf[150];
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// store error strings in flash to save RAM
#define error(s) sd.errorHalt_P(PSTR(s))
//------------------------------------------------------------------------------
// call back for file timestamps
void dateTime(uint16_t* date, uint16_t* time) {
    DateTime now = RTC.now();

  // return date using FAT_DATE macro to format fields
  *date = FAT_DATE(now.year(), now.month(), now.day());

  // return time using FAT_TIME macro to format fields
  *time = FAT_TIME(now.hour(), now.minute(), now.second());
}
//------------------------------------------------------------------------------
// format date/time
ostream& operator << (ostream& os, DateTime& dt) {
  os << dt.year() << '/' << int(dt.month()) << '/' << int(dt.day()) << ',';
  os << int(dt.hour()) << ':' << setfill('0') << setw(2) << int(dt.minute());
  os << ':' << setw(2) << int(dt.second()) << setfill(' ');
  return os;
}
//------------------------------------------------------------------------------


OneWire oneWire(3);
 
DallasTemperature sensors(&oneWire);

DeviceAddress temp1 = { 0x28, 0xD5, 0x3B, 0x28, 0x04, 0x00, 0x00, 0x37 };
DeviceAddress temp2 = { 0x28, 0xB1, 0x64, 0x28, 0x04, 0x00, 0x00, 0xD2 };
DeviceAddress temp3 = { 0x28, 0xB8, 0x72, 0x28, 0x04, 0x00, 0x00, 0xA4 };
DeviceAddress temp4 = { 0x28, 0xAB, 0x2B, 0x28, 0x04, 0x00, 0x00, 0xC0 };
DeviceAddress temp5 = { 0x28, 0x79, 0x9B, 0x28, 0x04, 0x00, 0x00, 0x25 };
DeviceAddress temp6 = { 0x28, 0xA6, 0x6C, 0x28, 0x04, 0x00, 0x00, 0x32 };
DeviceAddress temp7 = { 0x28, 0x27, 0x4B, 0x28, 0x04, 0x00, 0x00, 0x46 };
DeviceAddress temp8 = { 0x28, 0x90, 0x69, 0x28, 0x04, 0x00, 0x00, 0xBF };
DeviceAddress temp9 = { 0x28, 0x39, 0x3E, 0x28, 0x04, 0x00, 0x00, 0xB1 };
DeviceAddress temp10= { 0x28, 0x32, 0x31, 0x28, 0x04, 0x00, 0x00, 0x26 };
DeviceAddress temp11= { 0x28, 0x90, 0x57, 0x28, 0x04, 0x00, 0x00, 0x99 };
DeviceAddress temp12= { 0x28, 0x71, 0x76, 0x28, 0x04, 0x00, 0x00, 0xB2 };

void setup() {
  // For Leonardo
  while (!Serial) {}
  Serial.begin(9600);

  // pstr stores strings in flash to save RAM
  cout << endl << pstr("FreeRam: ") << FreeRam() << endl;

#if WAIT_TO_START
  cout << pstr("Type any character to start\n");
  while (Serial.read() < 0) {}
#endif  // WAIT_TO_START

  // connect to RTC
  if (!RTC.begin()) error("RTC failed");

  // set date time callback function
  SdFile::dateTimeCallback(dateTime);
  DateTime now = RTC.now();
  cout  << now << endl;

  // initialize the SD card
  if (!sd.begin(CHIP_SELECT)) sd.initErrorHalt();

  // create a new file in root, the current working directory
  char name[] = "LOGGER00.CSV";

  for (uint8_t i = 0; i < 100; i++) {
    name[6] = i/10 + '0';
    name[7] = i%10 + '0';
    if (sd.exists(name)) continue;
    logfile.open(name);
    break;
  }
  if (!logfile.is_open()) error("file.open");

  cout << pstr("Logging to: ") << name << endl;

  // format header in buffer
  obufstream bout(buf, sizeof(buf));

  bout << pstr("date,time");

  for (uint8_t i = 1; i <= 12; i++) {
    bout << pstr(",probe") << int(i);
  }
  
  logfile << buf << endl;

#if ECHO_TO_SERIAL
  cout << buf << endl;
#endif  // ECHO_TO_SERIAL

  sensors.begin();
  sensors.setResolution(temp1, 10);
  sensors.setResolution(temp2, 10);
  sensors.setResolution(temp3, 10);
  sensors.setResolution(temp4, 10);
  sensors.setResolution(temp5, 10);
  sensors.setResolution(temp6, 10);
  sensors.setResolution(temp7, 10);
  sensors.setResolution(temp8, 10);
  sensors.setResolution(temp9, 10);
  sensors.setResolution(temp10, 10);
  sensors.setResolution(temp11, 10);
  sensors.setResolution(temp12, 10);
}

//------------------------------------------------------------------------------
void loop() {
  uint32_t m;

  // use buffer stream to format line
  obufstream bout(buf, sizeof(buf));

  DateTime now = RTC.now();
  bout << now;
  
  sensors.requestTemperatures();
  bout << ',' << sensors.getTempC(temp1);
  bout << ',' << sensors.getTempC(temp2);
  bout << ',' << sensors.getTempC(temp3);
  bout << ',' << sensors.getTempC(temp4);
  bout << ',' << sensors.getTempC(temp5);
  bout << ',' << sensors.getTempC(temp6);
  bout << ',' << sensors.getTempC(temp7);
  bout << ',' << sensors.getTempC(temp8);
  bout << ',' << sensors.getTempC(temp9);
  bout << ',' << sensors.getTempC(temp10);
  bout << ',' << sensors.getTempC(temp11);
  bout << ',' << sensors.getTempC(temp12);
  bout << endl;

  // log data and flush to SD
  logfile << buf << flush;

  // check for error
  if (!logfile) error("write data failed");

#if ECHO_TO_SERIAL
  cout << buf;
#endif  // ECHO_TO_SERIAL

  delay(LOG_INTERVAL);

  // don't log two points in the same millis
  if (m == millis()) delay(1);
}
