
/***********************************************
 * RGB clock                                   *
 * Author: (c) Martijn van der Veen (Turiphro) *
 * License: GNU GPL, v2 or later               *
 * Version: 1.0.1                              *
 * Changelog:                                  *
 *  19/01/2012 Wrote code for RGB clock        *
 ***********************************************/

#include "Tlc5940.h"
#define TLC_MAX_PWM 4095
#include <Wire.h>
#define DS1307_I2C_ADDRESS 0x68


/* Settings */
boolean dimAtNight = true;
int intensity_night = TLC_MAX_PWM/50;
int intensity_day   = TLC_MAX_PWM/4;

boolean setDateTime = false; /* Set to true to set new date and time */
byte second  =  0, minute = 40, hour =  0;
byte day     = 23, month  =  1, year = 12;
byte weekDay =  1;


/* Global variables */
unsigned long startOfSecond = 0;
int intensity = intensity_day;


void setup()
{
  
  Serial.begin(9600);
  
  /* set up time (RTC DS1307) chip */
  Wire.begin();
  if (setDateTime) {
    setDateDs1307(second, minute, hour, weekDay, day, month, year);
  }
  
  /* set up PWM (TLC5940) chip */
  Tlc.init();
  
}


void loop()
{

  /* Update the real time each second */
  if (millis() - startOfSecond >= 1000) {
    startOfSecond = millis();
    getDateDs1307(&second, &minute, &hour, &weekDay, &day, &month, &year);
    /* for externally reading out the internal clock settings: */
    Serial.print(int(hour)); Serial.write(":"); Serial.print(minute); Serial.write(":"); Serial.print(second); Serial.write(" "); Serial.print(day); Serial.write("/"); Serial.print(month); Serial.write("/20"); Serial.print(year); Serial.println();
  }
  
  double t_sec   = double((millis() - startOfSecond))/1000;
  double t_10sec = double(int(second)%10)/10 + t_sec/10;   /* 1 second cycles are a bit flashy */
  double t_min   = double(second)/60 + t_sec/60;
  double t_hour  = double(minute)/60 + t_min/60;
  double t_day   = double(hour)/24 + t_hour/24;
  double t_week  = (double(weekDay) - 1.0)/7  + t_day/7;

  /* Decrease the intensity during evening and night, if desired. Change shift to change the peak. */
  if (dimAtNight) {
    double shift = 0.5; /* For 0.5: maximum at 12h, minimum at 24h */
    intensity = intensity_night \
              + (intensity_day-intensity_night)*0.5 \
              + (intensity_day-intensity_night)*0.5*cos( (t_day - shift) * 2 * PI );
  }
    
  Tlc.clear();
  /* linear RGB interpolation */
  double r, g, b;
  
  interpolate(t_week, &r, &g, &b);
  Tlc.set(0, int(intensity * r));
  Tlc.set(1, int(intensity * g));
  Tlc.set(2, int(intensity * b));
  
  interpolate(t_day, &r, &g, &b);
  Tlc.set(3, int(intensity * r));
  Tlc.set(4, int(intensity * g));
  Tlc.set(5, int(intensity * b));
  
  interpolate(t_hour, &r, &g, &b);
  Tlc.set(6, int(intensity * r));
  Tlc.set(7, int(intensity * g));
  Tlc.set(8, int(intensity * b));

  interpolate(t_min, &r, &g, &b);
  Tlc.set(9,  int(intensity * r));
  Tlc.set(10, int(intensity * g));
  Tlc.set(11, int(intensity * b));  
  
  //interpolate(t_sec, &r, &g, &b);
  interpolate(t_10sec, &r, &g, &b);
  Tlc.set(12, int(intensity * r));
  Tlc.set(13, int(intensity * g));
  Tlc.set(14, int(intensity * b));
  
  Tlc.update();

}



void interpolate(double t, double *r, double *g, double *b) {
  /* Linear interpolation
   *
   *    |\       /     |  /\          |     /\  
   * R: | \     /   G: | /  \      B: |    /  \ 
   *    |  \   /       |/    \        |   /    \
   *    ----------     ----------     ----------
   * With G and B peaks slightly shifted to the right,
   * such that red=(1,0,0) at t=0, 
   *        yellow=(0.5, 0.5, 0) at t=0.25,
   *         green=(0,1,0) at t=0.5,
   *          blue=(0,0,1) at t=0.75
   */
   
  *r = *g = *b = 0;
  if (t < 1.0/2) {
    *r = (1-2*t);
    *g = 2*t;
  } else if (t < 3.0/4) {
    *g = (1-4*(t-1.0/2));
    *b = 4*(t-1.0/2);
  } else {
    *b = (1-4*(t-3.0/4));
    *r = 4*(t-3.0/4);
  }
}


/* The following RTC helper functions are based on:
 * http://arduinotronics.blogspot.com/2010/10/ds1307-real-time-clock-working.html
 * http://www.instructables.com/id/The-Arduino-Weather-Station-Thermostat/step6/Arduino-Clock-Module/
 */

// Convert normal decimal numbers to binary coded decimal
byte decToBcd(byte val)
{
  return ( (val/10*16) + (val%10) );
}

// Convert binary coded decimal to normal decimal numbers
byte bcdToDec(byte val)
{
  return ( (val/16*10) + (val%16) );
}

// 1) Sets the date and time on the ds1307
// 2) Starts the clock
// 3) Sets hour mode to 24 hour clock
// Assumes you're passing in valid numbers
void setDateDs1307(byte second, // 0-59
                   byte minute, // 0-59
                   byte hour, // 0-23
                   byte dayOfWeek, // 1-7
                   byte dayOfMonth, // 1-28/29/30/31
                   byte month, // 1-12
                   byte year) // 0-99
{
  Wire.beginTransmission(DS1307_I2C_ADDRESS);
  Wire.write(byte(0));
  Wire.write(decToBcd(second)); // 0 to bit 7 starts the clock
  Wire.write(decToBcd(minute));
  Wire.write(decToBcd(hour));
  Wire.write(decToBcd(dayOfWeek));
  Wire.write(decToBcd(dayOfMonth));
  Wire.write(decToBcd(month));
  Wire.write(decToBcd(year));
  Wire.write(0x10); // sends 0x10 (hex) 00010000 (binary) to control register - turns on square wave
  Wire.endTransmission();
}

// Gets the date and time from the ds1307
void getDateDs1307(byte *second,
                   byte *minute,
                   byte *hour,
                   byte *dayOfWeek,
                   byte *dayOfMonth,
                   byte *month,
                   byte *year)
{
  // Reset the register pointer
  Wire.beginTransmission(DS1307_I2C_ADDRESS);
  Wire.write(byte(0x0));
  Wire.endTransmission();
  Wire.requestFrom(DS1307_I2C_ADDRESS, 7);
  // A few of these need masks because certain bits are control bits
  *second = bcdToDec(Wire.read() & 0x7f);
  *minute = bcdToDec(Wire.read());
  *hour = bcdToDec(Wire.read() & 0x3f); // Need to change this if 12 hour am/pm
  *dayOfWeek = bcdToDec(Wire.read());
  *dayOfMonth = bcdToDec(Wire.read());
  *month = bcdToDec(Wire.read());
  *year = bcdToDec(Wire.read());
}

