/*********************************************************
 **Code for the MechLabs Watch (MechWatch)
 **
 ** This code shows the time current time on the face of the watch
 **
 **   To display the time push any button when the face is blank.
 ** after a set amount of time the display will turn off to
 ** save battery power.
 **
 **   To set the time, push and hold a button until the display
 ** goes black and release the button. Push up/down increase and 
 ** decrease the hours. If the time is PM the 4 minute LEDs will 
 ** light up (not that it matters). Push and hold a button until
 ** the display goes black and release the button to switch to 
 ** changing the minutes. Push and hold a third time until the
 ** screen goes dark to save time. If the watch is left too long
 ** while setting the time the screen will go dark and any
 ** changes will be discarded
 * 
 **
 ** Copyright 2018
 ** Jonathan Hodgins
 *********************************************************/

#include "MechLabsWatch.h"
Watch watch;//create watch instance

#include <Wire.h>
#include "RTClib.h"
RTC_DS3231 rtc;//create real time clock instance

//enable code for deep sleep
#include <avr/power.h>
#include <avr/sleep.h>

/**********************************
 **state machine variables
 **
 **the state machine controls the display and setting the time
 **********************************/
int8_t userInput,oldUserInput;//current button state
int8_t buttonEvent = 0;//stores the user input
long buttonTimer = millis();//used to tell if the button is being held
long buttonHoldTime = 1400;//the mS to hold the button before something happens
long timeSetTimeout = 9000;//the mS to wait before the watch resets to default
long displayTime = 5500;//the length to display the time

//variables outlining all possible button states
#define NO_BUTTON_EVENT      0
#define BUTTON_EVENT_1       1
#define BUTTON_EVENT_1_HOLD  2
#define BUTTON_EVENT_2       3
#define BUTTON_EVENT_2_HOLD  4

byte state=0;//the current state of the watch

//the various states of the watch, each is described in more detatil in stateMachine()
#define DEFAULT_STATE        0
#define POWER_SAVER          1
#define EXIT_POWER_SAVER     2
#define EXIT_DEFAULT_STATE   3
#define ENTER_SET_TIME_STATE 4
#define SET_HOUR_STATE       5
#define INC_HOUR_STATE       6
#define DEC_HOUR_STATE       7
#define EXIT_HOUR_STATE      8
#define SET_MIN_STATE        9
#define INC_MIN_STATE        10
#define DEC_MIN_STATE        11
#define EXIT_MIN_STATE       12
#define ENTER_DEFAULT_STATE  13

/**********************************
 **clock variables
 **********************************/
int8_t hours;
int8_t minutes;
#define AM 0
#define PM 1
long lastClockTime=0, currentClockTime=0, lastTimeUpdated=0;//variables to keep the time going
boolean AMPM=0;//indicates if the current time is morning or afternoon, not dispalyed unless setting the time
long oneMin=60000;//the length of 1 minute in ms(adjusted for clock speed variances, you may also need to adjust)

/**********************************
 **input variables
 **********************************/
int8_t S1Pin=2, S2Pin=3;
boolean S1State,S2State;

boolean blinkVariable=0;//blinks at a regular interval to show when a time is being set
byte blinkRate = 500;//the mS the blink is on (and off)

/**********************************
 **code that runs once at startup
 **********************************/
void setup()
{
  //Initilize serial communications
  Serial.begin(115200);

  //take first input reading
  userInput = PIND&B00001100;

  //enable interrupts on switches
  pinMode(S1Pin, INPUT_PULLUP);//set switch 1 input
  pinMode(S2Pin, INPUT_PULLUP);//set switch 2 input
  //enabled and disables later in code, ignore these lines
  //attachInterrupt(digitalPinToInterrupt(S1Pin), S1Interrupt, FALLING);
  //attachInterrupt(digitalPinToInterrupt(S2Pin), S2Interrupt, FALLING);
  
  //Enable deep sleep
  set_sleep_mode(SLEEP_MODE_PWR_DOWN);
  sleep_enable();

  //RTC setup
  if (! rtc.begin())//initilize RTC and catch any errors
  {//if error
    Serial.println("Couldn't find RTC");
    while (1);
  }
  if (rtc.lostPower())//if the RTC lost power set the time to midnight
  {
    rtc.adjust(DateTime(2018,1,1,0,0,0));//Year,Month,Day,Hour,Min,Sec
  }

  //initilize watch instance
  watch.initilize();
}

/**********************************
 **The main program loop
 **********************************/
void loop()
{
  getUserInput();//check for button presses
  
  stateMachine();//decide what to do with the input and what to display
}


/**********************************
 **Checks the buttons and updates the variable
 ** buttonEvent with the results
 **********************************/
void getUserInput(void)
{
  //used to see if a button has been pushed or released,
  //which resets the holding timer. If the holding timer
  //eventually exceeds a set time the button is considered held
  oldUserInput = userInput;

  //read the buttons
  userInput = PIND&B00001100;
  S1State = (userInput&B00000100)>>2;
  S2State = (userInput&B00001000)>>3;

  //reset the timer if user input changes
  if(userInput != oldUserInput)
  {
    buttonTimer = millis();
  }

  //update the output
  if(!S1State)
  {
    buttonEvent=BUTTON_EVENT_1;
  }
  else if(!S2State)
  {
    buttonEvent=BUTTON_EVENT_2;
  }
  else
  {
    buttonEvent=NO_BUTTON_EVENT;
  }

  //change the result if the button has been held for a set length of time
  if(millis()-buttonTimer > buttonHoldTime)
  {
    if(!S1State)
      buttonEvent=BUTTON_EVENT_1_HOLD;
    if(!S2State)
      buttonEvent=BUTTON_EVENT_2_HOLD;
  }

}

/************************************
 * interrupt routines are unused, 
 * the interrupts are only used to
 * wake from sleep mode
 ***********************************/
//interrupt runs when switch 1 is pressed
void S1Interrupt(void)
{
  //getUserInput();//check for button presses
}
//interrupt runs when switch 2 is pressed
void S2Interrupt(void)
{
  //getUserInput();//check for button presses
}

/**********************************
 **This portion of the code decides what to do with 
 **the user input and what to show on the display
 **********************************/
void stateMachine(void)
{
  long stateTime=millis();
  //if nothing has happened for a while and the clock is not in the default state
  if(stateTime-buttonTimer > timeSetTimeout && state != DEFAULT_STATE && state != POWER_SAVER)
  {
    state = DEFAULT_STATE;//put it back in the default state
  }

  switch(state)
  {
  case DEFAULT_STATE:

    clockUpdate();//keep the time current
    
    watch.setTime(hours, minutes);//set the display to show the time
    watch.showTime();//show the current time
    
    if(stateTime-buttonTimer >= displayTime)
      state = POWER_SAVER;

    if(buttonEvent != NO_BUTTON_EVENT)//if a button is pushed
      state = EXIT_DEFAULT_STATE;//go check what to do
      
    break;
    
  case POWER_SAVER:
    //attach interrupts to wake from low power mode
    attachInterrupt(digitalPinToInterrupt(S1Pin), S1Interrupt, CHANGE);
    attachInterrupt(digitalPinToInterrupt(S2Pin), S2Interrupt, CHANGE);
    
    //enter sleep
    sleep_mode(); 
    
    //detach interrupts to stop them from constantly wasting cycles
    detachInterrupt(digitalPinToInterrupt(S1Pin));
    detachInterrupt(digitalPinToInterrupt(S2Pin));
    
    getUserInput();//see what woke the microcontroller
    
    if(buttonEvent != NO_BUTTON_EVENT)//if a button is pushed
      state = EXIT_POWER_SAVER;//leave power saver
    break;
    
  case EXIT_POWER_SAVER:
    if(buttonEvent == NO_BUTTON_EVENT)//wait for the user to release input
      state = DEFAULT_STATE;//change state to display the time
    break;

  case EXIT_DEFAULT_STATE:
    watch.setTime(hours, minutes);
    watch.showTime();
    if(buttonEvent == NO_BUTTON_EVENT)//if all buttons are released
      state = DEFAULT_STATE;//go back to the default state
    if(buttonEvent == BUTTON_EVENT_1_HOLD || buttonEvent == BUTTON_EVENT_2_HOLD)//if a button is held
        state = ENTER_SET_TIME_STATE;//set the time
    break;

  case ENTER_SET_TIME_STATE:
    if(buttonEvent == NO_BUTTON_EVENT)//wait for the user to release the button
      state = SET_HOUR_STATE;//change state to set the hour
    break;

  case SET_HOUR_STATE:
    updateBlink();//update the flashing variable for the display
    if(blinkVariable==HIGH)
    {
      if(hours>11)//turn on the 4 minute LEDs if it is PM
        watch.setTime(hours, 4);
      else//otherwise display the minutes
        watch.setTime(hours, minutes);
      watch.showTime();
    }
    if(buttonEvent == BUTTON_EVENT_1)//if button 1 is pushed
      state = INC_HOUR_STATE;//increase the hour
    if(buttonEvent == BUTTON_EVENT_2)//if button 2 is pushed
      state = DEC_HOUR_STATE;//decrease the hour
    break;

  case INC_HOUR_STATE:
    watch.setTime(hours, minutes);//update the display values
    watch.showTime();//display the stored time
    if(buttonEvent == NO_BUTTON_EVENT)//wait for the user to release the button
    {
      hours = (hours+1)%24;//increase the hour by one
      state = SET_HOUR_STATE;//and return to change the hour again
    }
    if(buttonEvent == BUTTON_EVENT_1_HOLD || buttonEvent == BUTTON_EVENT_2_HOLD)//if a button is held
      state = EXIT_HOUR_STATE;//go here to wait for the user and decide what state to go to next
    break;

  case DEC_HOUR_STATE:
    watch.setTime(hours, minutes);//update the display
    watch.showTime();//display the stored time
    if(buttonEvent == NO_BUTTON_EVENT)//wait for the user to release the button
    {
      hours = (hours-1);//decrease the hour by one
      if(hours < 0)
        hours=23;
      state = SET_HOUR_STATE;//and return to change the hour again
    }
    if(buttonEvent == BUTTON_EVENT_1_HOLD || buttonEvent == BUTTON_EVENT_2_HOLD)//if a button is held
      state = EXIT_HOUR_STATE;//go here to wait for the user and decide what state to go to next
    break;

  case EXIT_HOUR_STATE:
    if(buttonEvent == NO_BUTTON_EVENT)//wait for the user to release the button
    {
      state = SET_MIN_STATE;//and change state to set the minutes
      clockSet();//update the stored time in RTC
    }
    break;

  case SET_MIN_STATE:
    updateBlink();
    watch.setTime(hours, minutes);//update the display
    if(blinkVariable==HIGH)
      watch.showTime();
    if(buttonEvent == BUTTON_EVENT_1)//if button 1 is pushed
      state = INC_MIN_STATE;//increase the minutes
    if(buttonEvent == BUTTON_EVENT_2)//if button 2 is pushed
      state = DEC_MIN_STATE;//decrease the minutes
    break;

  case INC_MIN_STATE:
    watch.setTime(hours, minutes);//update the display
    watch.showTime();//display the stored time
    if(buttonEvent == NO_BUTTON_EVENT)//when all buttons are released
    {
      minutes = (minutes+1)%60;//increase the minutes by one
      state = SET_MIN_STATE;//and go back to change the minutes again
    }
    if(buttonEvent == BUTTON_EVENT_1_HOLD || buttonEvent == BUTTON_EVENT_2_HOLD)//if a button is held
      state = EXIT_MIN_STATE;//wait for the user to release in this state and decide what to do
    break;

  case DEC_MIN_STATE:
    watch.setTime(hours, minutes);//update the display
    watch.showTime();//display the stored time
    if(buttonEvent == NO_BUTTON_EVENT)//when all buttons are released
    {
      minutes = (minutes-1);//decrease the minutes by one
      if(minutes < 0)
        minutes=59;
      state = SET_MIN_STATE;//and go back to change the minutes again
    }
    if(buttonEvent == BUTTON_EVENT_1_HOLD || buttonEvent == BUTTON_EVENT_2_HOLD)//if a button is held
      state = EXIT_MIN_STATE;//wait for the user to release in this state and decide what to do
    break;

  case EXIT_MIN_STATE:
    if(buttonEvent == NO_BUTTON_EVENT)//when the button is released
    {
      state = ENTER_DEFAULT_STATE;//change state to set the alarm
      clockSet();//update the stored time in RTC
    }
    break;

  case ENTER_DEFAULT_STATE:
    if(buttonEvent == NO_BUTTON_EVENT)//wait until the buttons are released
    {
      state = DEFAULT_STATE;//and go back to the default state
      clockSet();//update the stored time in RTC
    }
    break;

  default://if something unexpected happens
    state = DEFAULT_STATE;//go back to the default state (time display)
    break;
  }
}

//toggle a variable to make the display blink while setting the time
void updateBlink(void)
{
  if(millis()%blinkRate>=blinkRate/2)
    blinkVariable=1;
  else
    blinkVariable=0;
}

//update the global variables minutes and hours with current values
void clockUpdate()
{
  DateTime now = rtc.now();
  minutes=now.minute();
  hours=now.hour();
}

//save new time to RTC chip
void clockSet()
{
  rtc.adjust(DateTime(2018,1,1,hours,minutes,0));//Year,Month,Day,Hour,Min,Sec
}

