/*
OpenFly.c

Author: Matt Dougan
Date: 10-17-2014

Notes for Programming the hex file
	-The microcontroller clk frequency is fairly low (128k), while by default many programmers will run at 1MHz. Reduce this to under 32kHz.


Overview of features:
	-built and tested on ATTINY24A
	-uses internal 128kHz osc for low power



*/ 

/* FUSE SETTINGS */
/*

-Select low speed, low power internal 128kHz oscillator
CKSEL3:0 = 0b0100

-Do not divide selected clock by 8
CKDIV8 = 0b0

BODSEL - disabled

-fast rising power
SUT1:0 = 0b01

*/



/* DEFINITONS */
	#define F_CPU			128000UL

	#define PWR_GOOD_MS		100

	#define PIN_PGOOD		PB0
	#define PIN_PBAD		PB1
	#define PULSE_LED_MS	400
	
	#define PIN_MOTOR		PA7
	
	#define PIN_HALL		PB2
	
	#define PIN_BATT_EN		PA1

/* INCLUDES */
	#include <avr/io.h>
	#include <util/delay.h>
	#include <avr/interrupt.h>
	#include <avr/power.h>
	#include <avr/sleep.h>
	#include <avr/wdt.h>
	

/* FUNCTION PROTOTYPES */
	void init_system();
	void config_pins();
	void reduce_power();
	void enter_sleep(uint8_t Sleep_Mode);
	void init_watchdog(uint8_t watchdog_time);
	void init_pwm();
	void stop_pwm();
	uint8_t battery_good();


/* VARIABLES */




/* MAIN */
int main(void)
{
    init_system();
	
	//It is very common for embedded programs to never exit from main. Additionally, they commonly have an infinite while loop!		
	while(1)
    {
        		
		//if zipper is not present, hall output is high and pin will read '1'
		if (PINB & _BV(PIN_HALL))
		{			
			power_timer0_enable();				//re-enable timer0 so we can use it
			init_pwm();							//start pwm (turn on motor)
			init_watchdog(WDTO_250MS);			//set sleep-time
			enter_sleep(SLEEP_MODE_IDLE);		//go to sleep (idle allows timer0 to continue running while processor sleeps)
			
			stop_pwm();							//stop pwm (turn off motor)
			power_timer0_disable();				//disable timer0 to save power
			init_watchdog(WDTO_250MS);
			enter_sleep(SLEEP_MODE_PWR_DOWN);
			
			power_timer0_enable();				//pulse the motor a second time
			init_pwm();
			init_watchdog(WDTO_250MS);
			enter_sleep(SLEEP_MODE_IDLE);
			
			stop_pwm();
			power_timer0_disable();			
		}
		
		//Max watchdog timeout is 8 seconds, so go to sleep twice since we don't want to buzz too often (unless you happen to like that)
		init_watchdog(WDTO_8S);
		enter_sleep(SLEEP_MODE_PWR_DOWN);
		init_watchdog(WDTO_8S);
		enter_sleep(SLEEP_MODE_PWR_DOWN);
		
    }
}



/* FUNCTIONS */

//Initialize system settings
void init_system()
{
	config_pins();
		
		
	//Check battery power
	//if good, pulse Pgood once		
	if(battery_good()) {
		PORTB |= _BV(PIN_PGOOD);
		_delay_ms(PULSE_LED_MS);
		PORTB &= ~_BV(PIN_PGOOD);				
	}
	else
	{
		while(1)	//go into infinite loop - blink until reset or battery replaced
		{
			PORTB &= ~_BV(PIN_PBAD);
			_delay_ms(PULSE_LED_MS);
			PORTB |= _BV(PIN_PBAD);
			_delay_ms(PULSE_LED_MS);			
		}		
	}
	//set pgood and pbad indicator back to input to save power
	PORTB &= ~(_BV(PIN_PGOOD)|_BV(PIN_PBAD));
	DDRB &= ~(_BV(PIN_PGOOD)|_BV(PIN_PBAD));
	
	//We finished some one-time start up things, so we can now turn off unused functionality
	reduce_power();	
	init_watchdog(WDTO_8S);	
	
}

//set i/o states
void config_pins()
{
	//Set up Power LEDs to output and initialize to off
	DDRB |= _BV(PIN_PGOOD)|_BV(PIN_PBAD);
	PORTB &= ~(_BV(PIN_PGOOD)|_BV(PIN_PBAD));	
	
	//Set up Battery measure enable and initialize to on
	DDRA |= _BV(PIN_BATT_EN);
	PORTA &= ~_BV(PIN_BATT_EN);
	
	//Set up Motor 
	DDRA |= _BV(PIN_MOTOR);
	PORTA &= ~_BV(PIN_MOTOR);
	
	//Set up Hall-Effect Reads
	DDRB &= ~_BV(PIN_HALL);
}

//We don't want to necessarily run the rest of the code if the battery voltage is too low to...
//A: get meaningful reads from the hall effect sensor
//B: run the vibrating motor
uint8_t battery_good()
{
	uint8_t power_good = 0;
	uint16_t result = 0;
		
	//We're going to do this a little bit more low level. Hang in there, it's just a datasheet, and bits...
	
	//Select internal reference and ADC0 (PA0)
	ADMUX = _BV(REFS1);		
	//let reference stabilize
	_delay_ms(10);
	
	//enable adc and start a conversion
	ADCSRA |= _BV(ADEN);
	ADCSRA |= _BV(ADSC);	
	
	//wait for ADC to complete conversion
	while (!(ADCSRA & _BV(ADIF)));	
	ADCSRA |= _BV(ADIF);
	
	//turn the battery enable back off once adc read is complete
	DDRA &= ~_BV(PIN_BATT_EN);
	
	result = ADCW;	
	if (result > 400)	//if battery voltage > 2.8V we're good
	{
		power_good = 1;
	}
		
	power_adc_disable();
	return power_good;
}


//Disables and stops unused modules to reduce power
//http://www.nongnu.org/avr-libc/user-manual/group__avr__power.html
void reduce_power()
{
	//disable brown out detector reference
	MCUCR |= _BV(BODSE);
	
	//DIDR0 |= 0xFF		//could use this to reduce power consumption of digital input buffers
	
	//disable unused portions
	power_adc_disable();
	power_timer0_disable();
	power_timer1_disable();
	power_usi_disable();
}


//Enter sleep, then wait for wakeup
//http://www.nongnu.org/avr-libc/user-manual/group__avr__sleep.html
void enter_sleep(uint8_t Sleep_Mode)
{
	//Set deepness of sleep
	set_sleep_mode(Sleep_Mode);
		
	cli();					//disable interrupts (writes to sleep control are timing sensitive)
	sleep_enable();			//enable sleep, which allows sleep to be entered
	sleep_bod_disable();	//disable brownout detector (i.e. power supply voltage dips too low)
	sei();					//re-enable interrupts
	sleep_cpu();			//go to sleep until an event wakes up cpu. Execution will resume here
	
	//after waking from sleep, disable sleep to avoid spurious activation
	sleep_disable();	
}



//This function sets the watchdog timer to expire in a set amount of time, triggering an interrupt to wake the processor
//http://www.nongnu.org/avr-libc/user-manual/group__avr__watchdog.html
void init_watchdog(uint8_t watchdog_time)
{

	wdt_disable();					//put watchdog in known, inactive state
	wdt_enable(watchdog_time);		//enable the watchdog timer with a preset time
	
	//enable watchdog interrupt
	WDTCSR |= _BV(WDIE);			//When watchdog timer completes, trigger an interrupt instead of a processor reset	
}


//Start Pulse Width Modulation to run the motor
void init_pwm()
{
	DDRA |= _BV(PIN_MOTOR);			//set port pin to output
	PORTA &= ~_BV(PIN_MOTOR);		
	
	//Set noninverting, fast PWM that counts from bottom to OCR0A. Selects main cpu clock as clock source.
	TCCR0A |= _BV(COM0B1)|_BV(WGM01)|_BV(WGM00);
	TCCR0B |= _BV(WGM02)|_BV(CS00);
	
	//We want to keep the PWM frequency pretty high. PWM_frequency = 128*10^3 / (OCR0A + 1)
		//(below 20kHz PWM, there's a chance you might hear an audible high frequency pitch - some energy is converted to sound)
	OCR0A = 5;
	
	//Sets duty cycle (percent of time motor is on) at (OCR0B + 1)/(OCR0A + 1)
	OCR0B = 5;	
}

void stop_pwm()
{
	//stop clock to Timer0 and therefore motor PWM
	TCCR0B &= _BV(CS02)|_BV(CS01)|_BV(CS00);
	
	//force motor into off-state
	PORTA &= ~_BV(PIN_MOTOR);
	DDRA &= ~_BV(PIN_MOTOR);
}



/* INTERRUPTS */
//http://www.nongnu.org/avr-libc/user-manual/group__avr__interrupts.html

//watchdog interrupt
ISR(WDT_vect)
{
	//re-enable watchdog interrupt - otherwise defaults back to reset behavior
	WDTCSR |= _BV(WDIE);	
	
}

//Catch any spurious interrupts, prevent reset
ISR(BADISR_vect)
{
	//occasionally an act of nature (radiation!) can cause interrupt flags to get set - this is very rare.
	//More likely a stack pointer was messed up, but having a catch-all interrupt is good practice (or useful for debug later).
	
}