#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

#include "twi/twi.h"
#include "usbconfig.h"
#include "usbdrv/usbdrv.h"

#define WIIEXT_TWI_ADDR 0x52 // this is the I2C slave address of any Wiimote extensions
#define SCROLL_TICK_THRESH 150 // controls speed of mouse wheel scrolling
#define ARROW_TICK_THRESH 150 // controls speed of arrow key pressing

// this data structure is described by the report descriptor
// see report descriptor below
static struct keyboard_report_t
{
	uint8_t report_id;
	uint8_t modifier; // bit flags for shift, ctrl, and alt, and other stuff
	uint8_t reserved; // useless for now
	uint8_t key[6];	  // HID keycodes
} keyboard_report;

static struct mouse_report_t
{
	uint8_t report_id;
	uint8_t buttons;   // button mask ( . . . . . M L R )
	int8_t x;		  // mouse x movement
	int8_t y;		  // mouse y movement
	int8_t v_wheel;	// mouse wheel movement
	int8_t h_wheel;	// mouse wheel movement
} mouse_report;

static struct gamepad_report_t
{
	uint8_t report_id;
	uint16_t buttons;
	int8_t left_x;
	int8_t left_y;
	int8_t right_x;
	int8_t right_y;
} gamepad_report;

PROGMEM char usbHidReportDescriptor[176] = {
// make sure the size matches USB_CFG_HID_REPORT_DESCRIPTOR_LENGTH in usbconfig.h

	// start of keyboard report descriptor
	0x05, 0x01,   // USAGE_PAGE (Generic Desktop)
	0x09, 0x06,   // USAGE (Keyboard)
	0xa1, 0x01,   // COLLECTION (Application)
	0x85, 0x01,   //   REPORT_ID (1)
	0x05, 0x07,   //   USAGE_PAGE (Keyboard)
	0x19, 0xe0,   //   USAGE_MINIMUM (Keyboard LeftControl)
	0x29, 0xe7,   //   USAGE_MAXIMUM (Keyboard Right GUI)
	0x15, 0x00,   //   LOGICAL_MINIMUM (0)
	0x25, 0x01,   //   LOGICAL_MAXIMUM (1)
	0x75, 0x01,   //   REPORT_SIZE (1)
	0x95, 0x08,   //   REPORT_COUNT (8)
	0x81, 0x02,   //   INPUT (Data,Var,Abs)
	0x95, 0x01,   //   REPORT_COUNT (1)
	0x75, 0x08,   //   REPORT_SIZE (8)
	0x81, 0x03,   //   INPUT (Cnst,Var,Abs)
	0x95, 0x05,   //   REPORT_COUNT (5)
	0x75, 0x01,   //   REPORT_SIZE (1)
	0x05, 0x08,   //   USAGE_PAGE (LEDs)
	0x19, 0x01,   //   USAGE_MINIMUM (Num Lock)
	0x29, 0x05,   //   USAGE_MAXIMUM (Kana)
	0x91, 0x02,   //   OUTPUT (Data,Var,Abs)
	0x95, 0x01,   //   REPORT_COUNT (1)
	0x75, 0x03,   //   REPORT_SIZE (3)
	0x91, 0x03,   //   OUTPUT (Cnst,Var,Abs)
	0x95, 0x06,   //   REPORT_COUNT (6)
	0x75, 0x08,   //   REPORT_SIZE (8)
	0x15, 0x00,   //   LOGICAL_MINIMUM (0)
	0x25, 0x65,   //   LOGICAL_MAXIMUM (101)
	0x05, 0x07,   //   USAGE_PAGE (Keyboard)
	0x19, 0x00,   //   USAGE_MINIMUM (Reserved (no event indicated))
	0x29, 0x65,   //   USAGE_MAXIMUM (Keyboard Application)
	0x81, 0x00,   //   INPUT (Data,Ary,Abs)
	0xc0,         // END_COLLECTION
	
	// start of mouse report descriptor
	0x05, 0x01,	   // USAGE_PAGE (Generic Desktop)
	0x09, 0x02,	   // USAGE (Mouse)
	0xa1, 0x01,	   // COLLECTION (Application)
	0x09, 0x01,	   //   USAGE (Pointer)
	0xa1, 0x00,	   //   COLLECTION (Physical)
	0x85, 0x02,	   //   REPORT_ID (2)
	0x05, 0x09,	   //	 USAGE_PAGE (Button)
	0x19, 0x01,	   //	 USAGE_MINIMUM (Button 1)
	0x29, 0x03,	   //	 USAGE_MAXIMUM (Button 3)
	0x15, 0x00,	   //	 LOGICAL_MINIMUM (0)
	0x25, 0x01,	   //	 LOGICAL_MAXIMUM (1)
	0x95, 0x03,	   //	 REPORT_COUNT (3)
	0x75, 0x01,	   //	 REPORT_SIZE (1)
	0x81, 0x02,	   //	 INPUT (Data,Var,Abs)
	0x95, 0x01,	   //	 REPORT_COUNT (1)
	0x75, 0x05,	   //	 REPORT_SIZE (5)
	0x81, 0x03,	   //	 INPUT (Cnst,Var,Abs)
	0x05, 0x01,	   //	 USAGE_PAGE (Generic Desktop)
	0x09, 0x30,	   //	 USAGE (X)
	0x09, 0x31,	   //	 USAGE (Y)
	0x09, 0x38,	   //	 USAGE (Wheel)
	0x15, 0x81,	   //	 LOGICAL_MINIMUM (-127)
	0x25, 0x7f,	   //	 LOGICAL_MAXIMUM (127)
	0x75, 0x08,	   //	 REPORT_SIZE (8)
	0x95, 0x03,	   //	 REPORT_COUNT (3)
	0x81, 0x06,	   //	 INPUT (Data,Var,Rel)
	0x05, 0x0c,	   //	 USAGE_PAGE (Consumer Devices)
	0x0a, 0x38, 0x02, //	 USAGE (Undefined)
	0x95, 0x01,	   //	 REPORT_COUNT (1)
	0x81, 0x06,	   //	 INPUT (Data,Var,Rel)
	0xc0,          //   END_COLLECTION
	0xc0,          // END_COLLECTION
	
	// start of gamepad report descriptor
	0x05, 0x01,	// USAGE_PAGE (Generic Desktop)
	0x09, 0x05,	// USAGE (Game Pad)
	0xa1, 0x01,	// COLLECTION (Application)
	0xa1, 0x00,	//   COLLECTION (Physical)
	0x85, 0x03, //   REPORT_ID (3)
	0x05, 0x09,	//	 USAGE_PAGE (Button)
	0x19, 0x01,	//	 USAGE_MINIMUM (Button 1)
	0x29, 0x10,	//	 USAGE_MAXIMUM (Button 16)
	0x15, 0x00,	//	 LOGICAL_MINIMUM (0)
	0x25, 0x01,	//	 LOGICAL_MAXIMUM (1)
	0x95, 0x10,	//	 REPORT_COUNT (16)
	0x75, 0x01,	//	 REPORT_SIZE (1)
	0x81, 0x02,	//	 INPUT (Data,Var,Abs)
	0x05, 0x01,	//	 USAGE_PAGE (Generic Desktop)
	0x09, 0x30,	//	 USAGE (X) // left X
	0x09, 0x31,	//	 USAGE (Y) // left Y
	0x09, 0x32,	//	 USAGE (Z) // right X
	0x09, 0x33,	//	 USAGE (Rx) // right Y
	0x15, 0x81,	//	 LOGICAL_MINIMUM (-127)
	0x25, 0x7f,	//	 LOGICAL_MAXIMUM (127)
	0x75, 0x08,	//	 REPORT_SIZE (8)
	0x95, 0x04,	//	 REPORT_COUNT (4)
	0x81, 0x02,	//	 INPUT (Data,Var,Abs)
	0xc0,		//   END_COLLECTION
	0xc0		// END_COLLECTION
};

static unsigned char idle_rate; // ignored
static unsigned char protocol_version = 0; // ignored

unsigned char usbFunctionSetup(unsigned char data[8])
{
	usbRequest_t *rq = (void *)data;

	// there are several request types but we only need to handle
	// 3 cases of the "class" type
	
	if ((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS) /* class request type */
	{
		if (rq->bRequest == USBRQ_HID_GET_REPORT) /* wValue: ReportType (highbyte), ReportID (lowbyte) */
		{
			// check report ID requested
			if (rq->wValue.bytes[0] == 1)
			{
				usbMsgPtr = &keyboard_report;
				return sizeof(keyboard_report);
			}
			else if (rq->wValue.bytes[0] == 2)
			{
				usbMsgPtr = &mouse_report;
				return sizeof(mouse_report);
			}
			else if (rq->wValue.bytes[0] == 3)
			{
				usbMsgPtr = &gamepad_report;
				return sizeof(gamepad_report);
			}
		}
		else if (rq->bRequest == USBRQ_HID_GET_IDLE)
		{
			usbMsgPtr = &idle_rate;
			return 1;
		}
		else if (rq->bRequest == USBRQ_HID_SET_IDLE)
		{
			idle_rate = rq->wValue.bytes[1];
		}
		else if (rq->bRequest == USBRQ_HID_GET_PROTOCOL)
		{
			usbMsgPtr = &protocol_version;
			return 1;
		}
		else if (rq->bRequest == USBRQ_HID_SET_PROTOCOL)
		{
			protocol_version = rq->wValue.bytes[1];
		}
		else if (rq->bRequest == USBRQ_HID_SET_REPORT)
		{
			if (rq->wLength.word == 1) // check data is available
			{
				return USB_NO_MSG; // send nothing but call usbFunctionWrite
			}
		}
	}
	else
	{
		// no vendor specific requests implemented
	}
	
	return 0;
}

// data from the computer is handled here
// this is actually going to be stuff like NUM LOCK, CAPS LOCK, and SCROLL LOCK LED data
unsigned char usbFunctionWrite(unsigned char *data, unsigned char len)
{
	// ignore this data
	return len;
}

// since we have set the encryption key to all-zero, we can predict how to decrypt it
// this may or may not be needed, with the clone controller I used, it's not needed
void wiiext_decrypt(unsigned char * array, int size)
{
	for (int i = 0; i < size; i++) {
		array[i] = (array[i] ^ 0x17) + 0x17;
	}
}

int main()
{	
	DDRC &= ~(_BV(5) | _BV(4)); // TWI pins as input
	PORTC |= _BV(5) | _BV(4); // enable pull-ups
	
	twi_init();
	sei();
	
	_delay_ms(25); // power up delay
	
	uint8_t twiBuffer[8];
	
	// initialize the Wii Classic Controller
	// make decryption predictable
	
	twiBuffer[0] = 0x40; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x46; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x4C; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 5, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	// retrieve center value of sticks
	
	twiBuffer[0] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
	twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
	//wiiext_decrypt(twiBuffer, 6);
	
	int centerLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
	int centerLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
	int centerRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
	int centerRightY = (int)((twiBuffer[2] & 0x1F) << 3);

	// large counters for feathering behaviour
	int ud_cnt = 0;
	int lr_cnt = 0;
	
	cli();
	
	usbInit(); // start v-usb
	usbDeviceDisconnect(); // enforce USB re-enumeration, do this while interrupts are disabled!
	_delay_ms(250);
	usbDeviceConnect();
	
	sei(); // enable interrupts
	
	while (1) // forever loop
	{
		usbPoll();
		
		// read raw values
		
		twiBuffer[0] = 0x00;
		twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
		twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
		//wiiext_decrypt(twiBuffer, 6);
		
		int rawLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
		int rawLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
		int rawRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
		int rawRightY = (int)((twiBuffer[2] & 0x1F) << 3);
		
		// increase these counters based on the amount of stick movement
		lr_cnt -= rawRightX - centerRightX;
		ud_cnt += rawRightY - centerRightY;
		
		int key_index = 0;
		
		// reset the report
		keyboard_report.modifier = 0;
		for (int i = 0; i < 6; i++) {
			keyboard_report.key[i] = 0;
		}
		
		if (bit_is_clear(twiBuffer[4], 5)) { // L for left-shift
			keyboard_report.modifier |= (1 << 1);
		}
		if (bit_is_clear(twiBuffer[5], 7)) { // ZL for left-CTRL
			keyboard_report.modifier |= (1 << 0);
		}
		if (bit_is_clear(twiBuffer[5], 2)) { // ZR for left-alt
			keyboard_report.modifier |= (1 << 2);
		}
		
		// if the counters pass the threshold, press the corresponding keys
		
		if (key_index < 6 && ud_cnt > ARROW_TICK_THRESH) { // up key from right stick
			ud_cnt = 0;
			keyboard_report.key[key_index] = 0x52;
			key_index++;
		}
		else if (key_index < 6 && ud_cnt < -ARROW_TICK_THRESH) { // down key from right stick
			ud_cnt = 0;
			keyboard_report.key[key_index] = 0x51;
			key_index++;
		}
		
		if (key_index < 6 && lr_cnt > ARROW_TICK_THRESH) { // left key from right stick
			lr_cnt = 0;
			keyboard_report.key[key_index] = 0x50;
			key_index++;
		}
		else if (key_index < 6 && lr_cnt < -ARROW_TICK_THRESH) { // right key from right stick
			lr_cnt = 0;
			keyboard_report.key[key_index] = 0x4F;
			key_index++;
		}
		
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 4)) { // minus
			keyboard_report.key[key_index] = 0x2D;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 2)) { // plus
			keyboard_report.key[key_index] = 0x57;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 3)) { // Home
			keyboard_report.key[key_index] = 0x4A;
			key_index++;
		}
		
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 0)) { // D-Pad Up = Page Up
			keyboard_report.key[key_index] = 0x4B;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 6)) { // D-Pad Down = Page Down
			keyboard_report.key[key_index] = 0x4E;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 1)) { // D-Pad Left = Backspace
			keyboard_report.key[key_index] = 0x2A;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 7)) { // D-Pad Right = Delete
			keyboard_report.key[key_index] = 0x4C;
			key_index++;
		}
		
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 1)) { // R for enter
			keyboard_report.key[key_index] = 0x28;
			key_index++;
		}
		
		keyboard_report.report_id = 1; // set report ID so computer knows what data struct is sent
		
		// wait until ready to send, then send it
		while (1) {
			usbPoll();
			if (usbInterruptIsReady())  {
				usbSetInterrupt((unsigned char *)(&keyboard_report), sizeof(keyboard_report));
				break;
			}
		}
		
		// use left stick for mouse movement
		mouse_report.x = (rawLeftX - centerLeftX) / 6;
		mouse_report.y = -(rawLeftY - centerLeftY) / 6;
		
		// use right stick or D-pad for scroll wheel
		mouse_report.h_wheel = 0;
		mouse_report.v_wheel = 0;
		
		mouse_report.buttons = 0;
		if (bit_is_clear(twiBuffer[5], 6) || bit_is_clear(twiBuffer[5], 5)) { // B or Y button as L
			mouse_report.buttons |= (1 << 0);
		}
		if (bit_is_clear(twiBuffer[5], 4)) { // A button as R
			mouse_report.buttons |= (1 << 1);
		}
		if (bit_is_clear(twiBuffer[5], 3)) { // X button as M
			mouse_report.buttons |= (1 << 2);
		}
		
		mouse_report.report_id = 2; // set report ID so computer knows what data struct is sent
		
		// wait until ready to send, then send it
		while (1) {
			usbPoll();
			if (usbInterruptIsReady())  {
				usbSetInterrupt((unsigned char *)(&mouse_report), sizeof(mouse_report));
				break;
			}
		}
		
		gamepad_report.left_x = rawLeftX - centerLeftX;
		gamepad_report.right_x = rawRightX - centerRightX;
		gamepad_report.left_y = rawLeftY - centerLeftY;
		gamepad_report.right_y = rawRightY - centerRightY;
		gamepad_report.buttons = ~((twiBuffer[4] << 8) | twiBuffer[5]);
		
		gamepad_report.report_id = 3; // set report ID so computer knows what data struct is sent
		
		// wait until ready to send, then send it
		while (1) {
			usbPoll();
			if (usbInterruptIsReady())  {
				usbSetInterrupt((unsigned char *)(&gamepad_report), sizeof(gamepad_report));
				break;
			}
		}
	}
	
	return 0;
}