#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

#include "twi/twi.h"
#include "usbconfig.h"
#include "usbdrv/usbdrv.h"

#define WIIEXT_TWI_ADDR 0x52 // this is the I2C slave address of any Wiimote extensions

// this data structure is described by the report descriptor
// see report descriptor below
static struct gamepad_report_t
{
	uint16_t buttons;
	int8_t left_x;
	int8_t left_y;
	int8_t right_x;
	int8_t right_y;
} gamepad_report;

PROGMEM char usbHidReportDescriptor[46] = {
// make sure the size matches USB_CFG_HID_REPORT_DESCRIPTOR_LENGTH in usbconfig.h
	0x05, 0x01,	// USAGE_PAGE (Generic Desktop)
	0x09, 0x05,	// USAGE (Game Pad)
	0xa1, 0x01,	// COLLECTION (Application)
	0xa1, 0x00,	//   COLLECTION (Physical)
	0x05, 0x09,	//	 USAGE_PAGE (Button)
	0x19, 0x01,	//	 USAGE_MINIMUM (Button 1)
	0x29, 0x10,	//	 USAGE_MAXIMUM (Button 16)
	0x15, 0x00,	//	 LOGICAL_MINIMUM (0)
	0x25, 0x01,	//	 LOGICAL_MAXIMUM (1) 
	0x95, 0x10,	//	 REPORT_COUNT (16)
	0x75, 0x01,	//	 REPORT_SIZE (1)
	0x81, 0x02,	//	 INPUT (Data,Var,Abs)
	0x05, 0x01,	//	 USAGE_PAGE (Generic Desktop)
	0x09, 0x30,	//	 USAGE (X) // left X
	0x09, 0x31,	//	 USAGE (Y) // left Y
	0x09, 0x32,	//	 USAGE (Z) // right X
	0x09, 0x33,	//	 USAGE (Rx) // right Y
	0x15, 0x81,	//	 LOGICAL_MINIMUM (-127)
	0x25, 0x7f,	//	 LOGICAL_MAXIMUM (127)
	0x75, 0x08,	//	 REPORT_SIZE (8)
	0x95, 0x04,	//	 REPORT_COUNT (4)
	0x81, 0x02,	//	 INPUT (Data,Var,Abs)
	0xc0,		//   END_COLLECTION
	0xc0		// END_COLLECTION
};

// not used for game pads
static unsigned char idleRate;

unsigned char usbFunctionSetup(unsigned char data[8])
{
	usbRequest_t *rq = (void *)data;

	// there are several request types but we only need to handle
	// 3 cases of the "class" type
	
	if ((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS) /* class request type */
	{
		if (rq->bRequest == USBRQ_HID_GET_REPORT) /* wValue: ReportType (highbyte), ReportID (lowbyte) */
		{			
			usbMsgPtr = &gamepad_report;
			return sizeof(gamepad_report);
		}
		else if (rq->bRequest == USBRQ_HID_GET_IDLE)
		{
			usbMsgPtr = &idleRate;
			return 1;
		}
		else if (rq->bRequest == USBRQ_HID_SET_IDLE)
		{
			idleRate = rq->wValue.bytes[1];
		}
	}
	else
	{
		// no vendor specific requests implemented
	}
	
	return 0;
}

// since we have set the encryption key to all-zero, we can predict how to decrypt it
// this may or may not be needed, with the clone controller I used, it's not needed
void wiiext_decrypt(unsigned char * array, int size)
{
	for (int i = 0; i < size; i++) {
		array[i] = (array[i] ^ 0x17) + 0x17;
	}
}

int main()
{	
	DDRC &= ~(_BV(5) | _BV(4)); // TWI pins as input
	PORTC |= _BV(5) | _BV(4); // enable pull-ups
	
	twi_init();
	sei();
	
	_delay_ms(25); // power up delay
	
	uint8_t twiBuffer[8];
	
	// initialize the Wii Classic Controller
	// make decryption predictable
	
	twiBuffer[0] = 0x40; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x46; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x4C; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 5, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	// retrieve center value of sticks
	
	twiBuffer[0] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
	twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
	//wiiext_decrypt(twiBuffer, 6);
	
	int centerLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
	int centerLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
	int centerRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
	int centerRightY = (int)((twiBuffer[2] & 0x1F) << 3);
	
	cli();
	
	usbInit(); // start v-usb
	usbDeviceDisconnect(); // enforce USB re-enumeration, do this while interrupts are disabled!
	_delay_ms(250);
	usbDeviceConnect();
	
	sei(); // enable interrupts
	
	while (1) // forever loop
	{
		usbPoll();
		
		// read raw values
		
		twiBuffer[0] = 0x00;
		twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
		twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
		//wiiext_decrypt(twiBuffer, 6);
		
		int rawLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
		int rawLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
		int rawRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
		int rawRightY = (int)((twiBuffer[2] & 0x1F) << 3);
		
		// calculate offset from original center position
		// set the values into the report
		gamepad_report.left_x = rawLeftX - centerLeftX;
		gamepad_report.right_x = rawRightX - centerRightX;
		gamepad_report.left_y = rawLeftY - centerLeftY;
		gamepad_report.right_y = rawRightY - centerRightY;
		gamepad_report.buttons = ~((twiBuffer[4] << 8) | twiBuffer[5]);
		
		// wait until endpoint is ready
		while (!usbInterruptIsReady()) {
			usbPoll();
		}
		
		// send report
		usbSetInterrupt((unsigned char *)(&gamepad_report), sizeof(gamepad_report));
	}
	
	return 0;
}