#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

#include "twi/twi.h"
#include "usbconfig.h"
#include "usbdrv/usbdrv.h"

#define WIIEXT_TWI_ADDR 0x52 // this is the I2C slave address of any Wiimote extensions
#define ARROW_TICK_THRESH 150 // controls speed of arrow key pressing

// this data structure is described by the report descriptor
// see report descriptor below
static struct keyboard_report_t
{
	uint8_t modifier; // bit flags for shift, ctrl, and alt, and other stuff
	uint8_t reserved; // useless for now
	uint8_t key[6];	  // HID keycodes
} keyboard_report;

PROGMEM char usbHidReportDescriptor[63] = {
// make sure the size matches USB_CFG_HID_REPORT_DESCRIPTOR_LENGTH in usbconfig.h
	0x05, 0x01,   // USAGE_PAGE (Generic Desktop)
    0x09, 0x06,   // USAGE (Keyboard)
    0xa1, 0x01,   // COLLECTION (Application)
    0x05, 0x07,   //   USAGE_PAGE (Keyboard)
    0x19, 0xe0,   //   USAGE_MINIMUM (Keyboard LeftControl)
    0x29, 0xe7,   //   USAGE_MAXIMUM (Keyboard Right GUI)
    0x15, 0x00,   //   LOGICAL_MINIMUM (0)
    0x25, 0x01,   //   LOGICAL_MAXIMUM (1)
    0x75, 0x01,   //   REPORT_SIZE (1)
    0x95, 0x08,   //   REPORT_COUNT (8)
    0x81, 0x02,   //   INPUT (Data,Var,Abs)
    0x95, 0x01,   //   REPORT_COUNT (1)
    0x75, 0x08,   //   REPORT_SIZE (8)
    0x81, 0x03,   //   INPUT (Cnst,Var,Abs)
    0x95, 0x05,   //   REPORT_COUNT (5)
    0x75, 0x01,   //   REPORT_SIZE (1)
    0x05, 0x08,   //   USAGE_PAGE (LEDs)
    0x19, 0x01,   //   USAGE_MINIMUM (Num Lock)
    0x29, 0x05,   //   USAGE_MAXIMUM (Kana)
    0x91, 0x02,   //   OUTPUT (Data,Var,Abs)
    0x95, 0x01,   //   REPORT_COUNT (1)
    0x75, 0x03,   //   REPORT_SIZE (3)
    0x91, 0x03,   //   OUTPUT (Cnst,Var,Abs)
    0x95, 0x06,   //   REPORT_COUNT (6)
    0x75, 0x08,   //   REPORT_SIZE (8)
    0x15, 0x00,   //   LOGICAL_MINIMUM (0)
    0x25, 0x65,   //   LOGICAL_MAXIMUM (101)
    0x05, 0x07,   //   USAGE_PAGE (Keyboard)
    0x19, 0x00,   //   USAGE_MINIMUM (Reserved (no event indicated))
    0x29, 0x65,   //   USAGE_MAXIMUM (Keyboard Application)
    0x81, 0x00,   //   INPUT (Data,Ary,Abs)
    0xc0          // END_COLLECTION
};

static unsigned char idle_rate; // ignored
static unsigned char protocol_version = 0; // ignored

unsigned char usbFunctionSetup(unsigned char data[8])
{
	usbRequest_t *rq = (void *)data;

	// there are several request types but we only need to handle
	// 3 cases of the "class" type
	
	if ((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS) /* class request type */
	{
		if (rq->bRequest == USBRQ_HID_GET_REPORT) /* wValue: ReportType (highbyte), ReportID (lowbyte) */
		{			
			usbMsgPtr = &keyboard_report;
			return sizeof(keyboard_report);
		}
		else if (rq->bRequest == USBRQ_HID_GET_IDLE)
		{
			usbMsgPtr = &idle_rate;
			return 1;
		}
		else if (rq->bRequest == USBRQ_HID_SET_IDLE)
		{
			idle_rate = rq->wValue.bytes[1];
		}
		else if (rq->bRequest == USBRQ_HID_GET_PROTOCOL)
		{
			usbMsgPtr = &protocol_version;
			return 1;
		}
		else if (rq->bRequest == USBRQ_HID_SET_PROTOCOL)
		{
			protocol_version = rq->wValue.bytes[1];
		}
		else if (rq->bRequest == USBRQ_HID_SET_REPORT)
		{
			if (rq->wLength.word == 1) // check data is available
			{
				return USB_NO_MSG; // send nothing but call usbFunctionWrite
			}
		}
	}
	else
	{
		// no vendor specific requests implemented
	}
	
	return 0;
}

// data from the computer is handled here
// this is actually going to be stuff like NUM LOCK, CAPS LOCK, and SCROLL LOCK LED data
unsigned char usbFunctionWrite(unsigned char *data, unsigned char len)
{
	// ignore this data
	return len;
}

// since we have set the encryption key to all-zero, we can predict how to decrypt it
// this may or may not be needed, with the clone controller I used, it's not needed
void wiiext_decrypt(unsigned char * array, int size)
{
	for (int i = 0; i < size; i++) {
		array[i] = (array[i] ^ 0x17) + 0x17;
	}
}

int main()
{	
	DDRC &= ~(_BV(5) | _BV(4)); // TWI pins as input
	PORTC |= _BV(5) | _BV(4); // enable pull-ups
	
	twi_init();
	sei();
	
	_delay_ms(25); // power up delay
	
	uint8_t twiBuffer[8];
	
	// initialize the Wii Classic Controller
	// make decryption predictable
	
	twiBuffer[0] = 0x40; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x46; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x4C; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 5, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	// retrieve center value of sticks
	
	twiBuffer[0] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
	twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
	//wiiext_decrypt(twiBuffer, 6);
	
	int centerLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
	int centerLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
	int centerRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
	int centerRightY = (int)((twiBuffer[2] & 0x1F) << 3);

	// large counters for feathering behaviour
	int ws_cnt = 0;
	int ad_cnt = 0;
	int ud_cnt = 0;
	int lr_cnt = 0;
	
	cli();
	
	usbInit(); // start v-usb
	usbDeviceDisconnect(); // enforce USB re-enumeration, do this while interrupts are disabled!
	_delay_ms(250);
	usbDeviceConnect();
	
	sei(); // enable interrupts
	
	while (1) // forever loop
	{
		usbPoll();
		
		// read raw values
		
		twiBuffer[0] = 0x00;
		twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
		twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
		//wiiext_decrypt(twiBuffer, 6);
		
		int rawLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
		int rawLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
		int rawRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
		int rawRightY = (int)((twiBuffer[2] & 0x1F) << 3);
		
		// increase these counters based on the amount of stick movement
		ad_cnt += rawLeftX - centerLeftX;
		ws_cnt += rawLeftY - centerLeftY;
		lr_cnt -= rawRightX - centerRightX;
		ud_cnt += rawRightY - centerRightY;
		
		int key_index = 0; // used to limit to 6 keycodes at a time
		
		// reset the report
		keyboard_report.modifier = 0;
		for (int i = 0; i < 6; i++) {
			keyboard_report.key[i] = 0;
		}
		
		if (bit_is_clear(twiBuffer[4], 5)) { // L for left-shift
			keyboard_report.modifier |= (1 << 1);
		}
		if (bit_is_clear(twiBuffer[5], 7)) { // ZL for left-CTRL
			keyboard_report.modifier |= (1 << 0);
		}
		if (bit_is_clear(twiBuffer[5], 2)) { // ZR for left-alt
			keyboard_report.modifier |= (1 << 2);
		}
		
		// if the counters pass the threshold, press the corresponding keys
		
		if (key_index < 6 && ad_cnt > ARROW_TICK_THRESH) { // D key from left stick
			ad_cnt = 0;
			keyboard_report.key[key_index] = 4 + 'd' - 'a';
			key_index++;
		}
		else if (key_index < 6 && ad_cnt < -ARROW_TICK_THRESH) { // A key from left stick
			ad_cnt = 0;
			keyboard_report.key[key_index] = 4 + 'a' - 'a';
			key_index++;
		}
		
		if (key_index < 6 && ws_cnt > ARROW_TICK_THRESH) { // W key from left stick
			ws_cnt = 0;
			keyboard_report.key[key_index] = 4 + 'w' - 'a';
			key_index++;
		}
		else if (key_index < 6 && ws_cnt < -ARROW_TICK_THRESH) { // S key from left stick
			ws_cnt = 0;
			keyboard_report.key[key_index] = 4 + 's' - 'a';
			key_index++;
		}
		
		if (key_index < 6 && ud_cnt > ARROW_TICK_THRESH) { // up key from right stick
			ud_cnt = 0;
			keyboard_report.key[key_index] = 0x52;
			key_index++;
		}
		else if (key_index < 6 && ud_cnt < -ARROW_TICK_THRESH) { // down key from right stick
			ud_cnt = 0;
			keyboard_report.key[key_index] = 0x51;
			key_index++;
		}
		
		if (key_index < 6 && lr_cnt > ARROW_TICK_THRESH) { // left key from right stick
			lr_cnt = 0;
			keyboard_report.key[key_index] = 0x50;
			key_index++;
		}
		else if (key_index < 6 && lr_cnt < -ARROW_TICK_THRESH) { // right key from right stick
			lr_cnt = 0;
			keyboard_report.key[key_index] = 0x4F;
			key_index++;
		}
		
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 4)) { // minus
			keyboard_report.key[key_index] = 0x2D;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 2)) { // plus
			keyboard_report.key[key_index] = 0x57;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 3)) { // Home
			keyboard_report.key[key_index] = 0x4A;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 6)) { // B
			keyboard_report.key[key_index] = 4 + 'b' - 'a';
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 5)) { // Y
			keyboard_report.key[key_index] = 4 + 'y' - 'a';
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 4)) { // A
			keyboard_report.key[key_index] = 4 + 'a' - 'a';
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 3)) { // X
			keyboard_report.key[key_index] = 4 + 'x' - 'a';
			key_index++;
		}
		
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 0)) { // D-Pad Up = Page Up
			keyboard_report.key[key_index] = 0x4B;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 6)) { // D-Pad Down = Page Down
			keyboard_report.key[key_index] = 0x4E;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[5], 1)) { // D-Pad Left = Backspace
			keyboard_report.key[key_index] = 0x2A;
			key_index++;
		}
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 7)) { // D-Pad Right = Delete
			keyboard_report.key[key_index] = 0x4C;
			key_index++;
		}
		
		if (key_index < 6 && bit_is_clear(twiBuffer[4], 1)) { // R for enter
			keyboard_report.key[key_index] = 0x28;
			key_index++;
		}
		
		// wait until ready to send
		while (!usbInterruptIsReady()) {
			usbPoll();
		}
		
		// send it
		usbSetInterrupt((unsigned char *)(&keyboard_report), sizeof(keyboard_report));
	}
	
	return 0;
}