/*********************************************************************
 * Weather 5 receiving data from NRF24L01 module
 * and displaying them on serial
 * MTSuzuki
 Copyright (C) 2012 James Coliz, Jr. <maniacbug@ymail.com>
 */

#include <RF24Network.h>
#include <RF24.h>
#include <SPI.h>

#include "DHT.h"

unsigned long startTime;
// clearTime will reset data to make sure nrf24L01 modules still sending
unsigned long clearTime = 120000;	// in milliseconds

// nRF24L01(+) radio (CE, CSN) attached 
RF24 radio(9,10);
// Network uses that radio
RF24Network network(radio);

// Channel of our node
const uint16_t channel = 30;
// Address of our node
const uint16_t this_node = 0;

DHT dht;
#define DHTPin 4

// Structure of our payloads
struct payload_t1{			// 32 bytes max
  unsigned long counter;	// 4 bytes
  double TempC;				// 4 bytes
  double RH;				// 4 bytes
  double Vcc;				// 4 bytes
  double BP;				// 4 bytes
} payload1;
struct payload_t2{
  double RainAccum;			// 4 bytes 
  double MPH;				// 4 bytes
  double GustMPH;			// 4 bytes
  double WDIR; 				// 4 bytes
} payload2;

unsigned long Counter;
double ExtTempC;
double ExtTempF;
double ExtRH;
double ExtVcc;
double ExtBP;
double ExtRainAccum;
double MPH;
double GustMPH;
double WDIR;

double twc;		//Temperature wind chill
double IntTempC;
double IntRH;

//Prototypes for utility functions
void getEnvironmentals();	// getEnvironmentals
void sendSerial();			// send Serial data
void getRadioData();		// get Radio data
void clearTemperatures();	// clear temps to make sure modules are active
void getWindChill();		// get wind chill

void setup(){

//	dht.setup(DHTPin);

	Serial.begin(115200);
 
	SPI.begin();
	// Radio setup
	radio.begin();
	 // optionally, increase the delay between retries & # of retries
	radio.setRetries(15,15);
	
	// network.begin(/*channel*/, /*node address*/);
	network.begin(channel, this_node);
	
	startTime = millis();		// start the timer

}

void loop() {

	delay(2000);	//Power up delay
	
	// Pump the radio network regularly
	network.update();
	
	getEnvironmentals();	
	// Is there anything ready for us?
	while ( network.available() ){
		// If so, grab it and print it out
		getRadioData();
	}

	sendSerial();
	delay(58000);
//	delay(5000);
	if ( (millis() - startTime) > clearTime){ 
		clearTemperatures();  
		startTime = millis();		// restart the timer
	}
}

//////////////////////////////////////////////////////////////////
// getEnvironmentals
//////////////////////////////////////////////////////////////////
void getEnvironmentals()
{
	IntTempC = dht.getTemperature();
	if (isnan(IntTempC)) IntTempC = -100;
	IntRH = dht.getHumidity();
	if (isnan(IntRH)) IntRH = -100;
}

//////////////////////////////////////////////////////////////////////////////////
// sendSerial();			// send to serial port
//////////////////////////////////////////////////////////////////////////////////
void sendSerial(){
	//Serial.println("$, ExtTempC, ExtRH, ExtVCC, ExtBP, ExtRainAccum, MPH, GustMPH, WDIR
	Serial.print("$");
	Serial.print(ExtTempF, 1);
	Serial.print(", ");
	Serial.print(ExtRH, 1);
	Serial.print(", ");
	Serial.print(ExtVcc, 1);
	Serial.print(", ");
	Serial.print(ExtBP, 2);
	Serial.print(", ");
	Serial.print(ExtRainAccum);
	Serial.print(", ");
	Serial.print(MPH, 1);
	Serial.print(", ");
	Serial.print(GustMPH, 1);
	Serial.print(", ");
	Serial.print(WDIR, 1);
	Serial.print(", ");
	Serial.print(twc, 1);
	Serial.print(", ");
	Serial.println(Counter, 1);
/*	
	Serial.print(2);
	Serial.print(" ");
	Serial.print(IntTempC, 1);
	Serial.print(" ");
	Serial.println(IntRH, 1);
*/
}

//////////////////////////////////////////////////////////////////////////////////
// getRadioData()					// get Network data
//////////////////////////////////////////////////////////////////////////////////
void getRadioData(){
	RF24NetworkHeader header;
	bool done1 = false;
    bool done2 = false;  
	while (!done1 && !done2){

		done1 = network.read(header,&payload1, sizeof(payload1));
		Counter = payload1.counter;
		ExtTempC = payload1.TempC;
		ExtRH = payload1.RH;
		ExtVcc = payload1.Vcc;
		ExtBP = payload1.BP;

		done2 = network.read(header,&payload2, sizeof(payload2));
		
		ExtRainAccum = payload2.RainAccum;
		MPH = payload2.MPH;
		GustMPH = payload2.GustMPH;
		WDIR = payload2.WDIR;
	}
	if (done1 && done2){
		getWindChill();
		sendSerial();
	}
	
}
//////////////////////////////////////////////////////////////////////////////////
// getWindChill()	// get wind chill
//////////////////////////////////////////////////////////////////////////////////

void getWindChill(){		// get wind chill
	ExtTempF = ExtTempC * 9.0/5.0 +32.0;
	if ((ExtTempF <= 50.0) && (MPH >3.0)){
	// U.S. National Weather Service 2001
		twc = 35.74 + (0.6215 * ExtTempF) - (35.75 * pow(MPH, 0.16)) + (0.4275 * ExtTempF * pow(MPH, 0.16));
	}
	else	twc = ExtTempF;
}
//////////////////////////////////////////////////////////////////////////////////
// clearTemperatures()					// clear temps to make sure modules are active
//////////////////////////////////////////////////////////////////////////////////

void clearTemperatures(){
		ExtTempC = 0.0;
		ExtRH = 0.0;
		ExtVcc = 0.0;
		ExtBP = 0.0;
		ExtRainAccum = 0.0;
		MPH = 0.0;
		GustMPH = 0.0;
}
