/* A Weather station data logger for the Arduino
 * Creator: LOG Michael Suzuki
 * 2016
 * Uses US1881 Hall effect latch for Rain and for wind speed
 * Transmit data via nrf24L01

*/
/***************************************************************************
  This is a library example for the HMC5883 magnentometer/compass

  Designed specifically to work with the Adafruit HMC5883 Breakout
  http://www.adafruit.com/products/1746
 
  *** You will also need to install the Adafruit_Sensor library! ***

  These displays use I2C to communicate, 2 pins are required to interface.

  Adafruit invests time and resources providing this open source code,
  please support Adafruit andopen-source hardware by purchasing products
  from Adafruit!

  Written by Kevin Townsend for Adafruit Industries with some heading example from
  Love Electronics (loveelectronics.co.uk)
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the version 3 GNU General Public License as
 published by the Free Software Foundation.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

 ***************************************************************************/

/*************************************************** 
This sketch shows how to use the SFE_BMP180 library to read the
Bosch BMP180 barometric pressure sensor.
https://www.sparkfun.com/products/11824
 ****************************************************/
//  Copyright (C) 2012 James Coliz, Jr. <maniacbug@ymail.com>

#include <Wire.h>
#include "DHT.h"
#include <SFE_BMP180.h>
#include <Adafruit_Sensor.h>
#include <Adafruit_HMC5883_U.h>

/* Assign a unique ID to this sensor at the same time */
Adafruit_HMC5883_Unified mag = Adafruit_HMC5883_Unified(12222);

#define ALTITUDE 1417.0 // Altitude meters, Loma

// nrf24L01 stuff
#include <RF24Network.h>
#include <RF24.h>
#include <SPI.h>

// nRF24L01(+) radio attached  (CE, CSN)
RF24 radio(9,10);
// Network uses that radio
RF24Network network(radio);
// Channel of our node

const uint16_t channel = 30;

// CHANGE THIS TO CORRECT NODE
const uint16_t this_node = 1;
// CHANGE THIS TO CORRECT NODE
// Address of the mother node
const uint16_t other_node = 0;

// How many packets have we sent already
unsigned long packets_sent;

// Structure of our payload, limited to 32 bytes
struct payload_t1			
{
  unsigned long counter;	// 4 bytes
  double TempC;				// 4 bytes
  double RH;				// 4 bytes
  double Vcc;				// 4 bytes
  double BP;				// 4 bytes
} payload1;
struct payload_t2
{
  double RainAccum;			// 4 bytes 
  double MPH;				// 4 bytes
  double GustMPH;			// 4 bytes
  double WDIR; 				// 4 bytes
} payload2;

// Analog pins
#define BatPin 0

// Digital pins
#define greenLEDpin 3
#define RainPin 7
#define WindSpeedPin 2
#define DHTPin 4

// Dht22 variables
DHT dht;	//DHT22 object
double TempC;
double RH;

// BMP180 variables
SFE_BMP180 pressure;	// BMP180 object
char status;
double T,P,p0,a;
double BP;	//Barometric pressure

// Battery variable
double BatVoltage;	// Battery voltage

// Wind variables
bool WindHigh = false;
int CntRPM = 0;
//MinPeriod is time for one revolution in milliseconds
// This represents gust speed
double MinPeriod = 100000.0;	// arbitrarily large number
double StartPeriod = 0.0;
double RPM = 0.0;
double MPH =0.0;
double GustMPH=0.0;

//WDIR is wind direction reading from 0.0 to 360.0
double WDIR;

//Rain variables
bool RainHigh = false;
const double LowAmt = 5.0;	//when rain is low, takes this ml to trip
const double HiAmt = 5.0;	//when rain is high, takes this ml to trip
double RainAccum = -5.0;	//Rain accumulator since start of sample

unsigned long LogInterval = 60000;	// milliseconds 60 seconds
//unsigned long LogInterval = 6000;	// test value

unsigned long StartTime;		//Log Interval counter
const byte NUM_SAMPLES = 10;	// Number of samples for analog reads

//Prototypes for utility functions
double getBatVolt();			// Battery voltage calculated
double getWDirection();			// Wind direction calculated
void calculateWind();			// Calculate wind speed and gust
void getEnvironmentals();		// get temp, rh, bp
void sendPayload();				// check if time to send payload
void Display();
 
void setup(void) {
	int dummy;
	analogReference(EXTERNAL); 		//External reference 3.3V
	dummy = analogRead(BatPin); 	//discard first analogRead
	Serial.begin(115200);

	pinMode(RainPin, INPUT); 
	pinMode(WindSpeedPin, INPUT); 

  /* Initialize the sensor */
  if(!mag.begin())
  {
    /* There was a problem detecting the HMC5883 ... check your connections */
    Serial.println("Ooops, no HMC5883 detected ... Check your wiring!");
    while(1);
  }
  mag.setMagGain(HMC5883_MAGGAIN_8_1);
  
	// DHT 22 setup
 	dht.setup(DHTPin);
	// BMP180
	pressure.begin();	
	// Radio stuff nrf24L01
	SPI.begin();
	radio.begin();
	 // optionally, increase the delay between retries & # of retries
	radio.setRetries(15,15);

	network.begin(channel, this_node);
	
// Power down the radio. Note that the radio will get powered back up on the next powerUp() call.
	radio.powerDown();
	
	StartTime=millis(); //Initialize StartTime
}
void loop(void){
	// Wind calculator, looks for RPM continuously
	// Look for High
	if (digitalRead(WindSpeedPin) == HIGH){
		WindHigh = true;
	}
   //Look for Low thus a High to Low transition
   else if ((WindHigh == true)&&(digitalRead(WindSpeedPin) == LOW)){
		//Increment counter
		CntRPM++;
		WindHigh = false;
		// See if this is least MinPeriod, highest Gust
		if (StartPeriod != 0.0){ //Not first sample
			if ((millis()-StartPeriod) < MinPeriod){
				MinPeriod = millis()-StartPeriod;
			}
		}
		StartPeriod = millis();
   }
   
   // Rain calculator, looks for Rain continuously
   // Look for low to high
   if ((RainHigh == false)&&(digitalRead(RainPin) == HIGH)){
		RainHigh = true;
		RainAccum += LowAmt;
	} // or high to low
   if ((RainHigh == true)&&(digitalRead(RainPin) == LOW)){
		RainHigh = false;
		RainAccum += HiAmt;
    }

	if ((millis() - StartTime) > LogInterval){  // Log Interval has passed
		radio.powerUp();
		network.update();
		getEnvironmentals();
		BatVoltage=getBatVolt();
		WDIR = getWDirection();
		calculateWind();
		Display();
		sendPayload();
		
		StartTime = millis();	//Get StartTime for next LogInterval
		// Clear variables for next LogInterval
		StartPeriod = 0.0;
		MinPeriod = 100000.0; 
		WindHigh = false;
		CntRPM = 0;
		RainAccum = 0.0;
	}
}
//////////////////////////////////////////////////////////////////////////////////
// Read analog input for BatPin averaged over NUM_SAMPLES
// Uses a running average
// Vcc is scaled with a voltage divider * 300K/(91K + 300K) so reverse
// Should multiply be 1.3
//////////////////////////////////////////////////////////////////////////////////
double getBatVolt(){
	double SumTotal=0;
	for (byte j=0;j<NUM_SAMPLES;j++){    
		SumTotal+= analogRead(BatPin);
		delay(10);
	}    		
	return  ((SumTotal/NUM_SAMPLES)*3.24/1023.0)*1.3;
}	

//////////////////////////////////////////////////////////////////////////////////
// Read analog input for WindDirectionPin averaged over NUM_SAMPLES
// Uses a running average
//////////////////////////////////////////////////////////////////////////////////

double getWDirection(){
	/* Get a new sensor event */
	sensors_event_t event; 
	mag.getEvent(&event);
	float heading = atan2(event.magnetic.y, event.magnetic.x);
  
	// Correct for when signs are reversed.
	if(heading < 0)
		heading += 2*PI;
    
	// Check for wrap due to addition of declination.
	if(heading > 2*PI)
		heading -= 2*PI;
   
	// Convert radians to degrees for readability.
	float headingDegrees = heading * 180/M_PI; 
  		
	return  (headingDegrees);
}
//////////////////////////////////////////////////////////////////
// calculateWind speed and Gust
//////////////////////////////////////////////////////////////////
void calculateWind(){
	RPM = CntRPM * (60.0 / (LogInterval/1000));
	MPH = RPM * 0.054;	// Calibration value
	// GustMPH is calculated = 60 * 0.054/Period in seconds
	//GustMPH = 60.0 * 0.054 / (MinPeriod/1000)
	GustMPH = 60000.0 * 0.054 / MinPeriod;
}

//////////////////////////////////////////////////////////////////
// getEnvironmentals
//////////////////////////////////////////////////////////////////
void getEnvironmentals(){
	
	TempC = dht.getTemperature();
	// Update only if valid number
	if (isnan(TempC)) TempC = -100.0;
	RH = dht.getHumidity();
	if (isnan(RH)) RH = -100.0;

	
	//get BMP180 pressure
  status = pressure.startTemperature();
  if (status != 0){
    // Wait for the measurement to complete:
    delay(status);
    // Retrieve the completed temperature measurement:
    status = pressure.getTemperature(T);
    if (status != 0){
      // Start a pressure measurement:
      // If request is successful, the number of ms to wait is returned.
      status = pressure.startPressure(3);
      if (status != 0){
        // Wait for the measurement to complete:
        delay(status);
        // Retrieve the completed pressure measurement:
        status = pressure.getPressure(P,T);
        if (status != 0){
          // The pressure sensor returns abolute pressure, which varies with altitude.
          // Parameters: P = absolute pressure in mb, ALTITUDE = current altitude in m.
          // Result: p0 = sea-level compensated pressure in mb
          p0 = pressure.sealevel(P,ALTITUDE);
		  BP = p0*0.0295333727;
        }
        else Serial.println("error retrieving pressure measurement\n");
      }
      else Serial.println("error starting pressure measurement\n");
    }
    else Serial.println("error retrieving temperature measurement\n");
  }
  else Serial.println("error starting temperature measurement\n");
}
//////////////////////////////////////////////////////////////////////////////////
// sendPayload();					// send payloads
//////////////////////////////////////////////////////////////////////////////////
void sendPayload(){
	bool ok;
	// limits counter to 10,000
	if (packets_sent > 10000) packets_sent = 0;
	payload1 = { packets_sent++, TempC, RH, BatVoltage, BP };
	payload2 = {RainAccum, MPH, GustMPH, WDIR };
	RF24NetworkHeader header(/*to node*/ other_node);
	
	ok = network.write(header,&payload1,sizeof(payload1));
	
	ok = network.write(header,&payload2,sizeof(payload2));
	
// Power down the radio. Note that the radio will get powered back up on the next powerUp() call.
	radio.powerDown();
}

void Display(){
	Serial.print("Batt voltage: ");
	Serial.print(BatVoltage,2);
	Serial.print(" TempC: ");
	Serial.print(TempC,2);
	Serial.print(" RH: ");
	Serial.print(RH,1);
	Serial.print(" BP: ");
    Serial.print(BP,2);
	Serial.print(" Rain: ");
    Serial.print(RainAccum,2);
	Serial.print(" Speed: ");
    Serial.print(MPH, 1);
	Serial.print(" Gust: ");
    Serial.print(GustMPH, 1);
	Serial.print(" Dir: ");
    Serial.print(WDIR,1);
	Serial.println();
}

