//  Copyright (C) 2012 James Coliz, Jr. <maniacbug@ymail.com>
// Narcoleptic - A sleep library for Arduino
// Copyright (C) 2010 Peter Knight (Cathedrow)

/**
 * L.O.G. measuring temperature with a DS18B20
 * and measuring Vcc 
 * For each node, change DS18B20 DeviceAddress
 * and this_node address
 * uses Narcoleptic library to put Arduino to sleep
 */

#include <RF24Network.h>
#include <RF24.h>
#include <SPI.h>
#include <OneWire.h>
#include <DallasTemperature.h>
#include <Narcoleptic.h>

int sleepDelay = 10000;	// in milliseconds

// DS18B20 is plugged into Arduino D4 
#define ONE_WIRE_BUS 4
// Setup a oneWire instance to communicate with any OneWire devices
OneWire oneWire(ONE_WIRE_BUS);
// Pass our oneWire reference to Dallas Temperature. 
DallasTemperature sensors(&oneWire);
// This is DS18B20 #1 address
DeviceAddress Therm1 = { 0x28, 0x44, 0xEC, 0xB2, 0x04, 0x00, 0x00, 0x44 };
float temp1C;
float temp1F;

// nRF24L01(+) radio attached  (CE, CSN)
RF24 radio(9,10);
// Network uses that radio
RF24Network network(radio);
// Channel of our node
const uint16_t channel = 90;
// Address of our node
const uint16_t this_node = 1;
// Address of the other node
const uint16_t other_node = 0;

// How many packets have we sent already
unsigned long packets_sent;

// Structure of our payload, limited to 32 bytes
struct payload_t			// 32 bytes max
{
  unsigned long counter;	// 4 bytes
  float tempC;				// 4 bytes
  float tempF;				// 4 bytes
  float Vcc;				// 4 bytes
};

// Analog pins
#define VccPin 0
// Digital pins
#define ActivePin 7

float Vcc; 	//Supplied Voltage

// Prototypes
void getTemperature(DeviceAddress);	// getTemperature
void getVoltage();					// getVoltage
void sendPayload();					// check if time to send payload

void setup(void)
{
	analogReference(INTERNAL); 		// Set analog reference to 1.1V
	analogRead(VccPin); 			//discard first analogRead
	pinMode(ActivePin, OUTPUT);		// Set for output
  
	// Start up the library
	sensors.begin();
	// set the DS18B20 resolution 
	sensors.setResolution(Therm1, 12);
	
	SPI.begin();
	radio.begin();
	network.begin(channel, this_node);

	// Power down the radio. Note that the radio will get powered back up on the next write() call.
	radio.powerDown();
}

void loop(void){
	// Pump the network regularly
	network.update();
	getVoltage();
	getTemperature(Therm1);
	sendPayload();
	Narcoleptic.delay(sleepDelay); // During this time power consumption is minimized
}

//////////////////////////////////////////////////////////////////
// getTemperature
//////////////////////////////////////////////////////////////////
void getTemperature(DeviceAddress deviceAddress)
{
	sensors.requestTemperatures();
	temp1C = sensors.getTempC(deviceAddress);
	temp1F = sensors.getTempF(deviceAddress);
}

//////////////////////////////////////////////////////////////////////////////////
// Read analog input for VccPin averaged over NUM_SAMPLES
// Uses a running average
// Vcc is scaled with a voltage divider * 75K/(75K + 240K) so reverse
// Should be 4.2, try 3.9
//////////////////////////////////////////////////////////////////////////////////
void getVoltage(){
	const byte NUM_SAMPLES = 20;
	float SumTotal=0;
	for (byte j=0;j<NUM_SAMPLES;j++){    
		SumTotal+= analogRead(VccPin);
		delay(10);
	}    		
	Vcc =  ((SumTotal/NUM_SAMPLES)*1.1/1023.0)*3.9;
}
//////////////////////////////////////////////////////////////////////////////////
// sendPayload();					// send payload
//////////////////////////////////////////////////////////////////////////////////
void sendPayload(){
	digitalWrite(ActivePin, HIGH);		// Turn on LED
	payload_t payload = { packets_sent++, temp1C, temp1F, Vcc };
	RF24NetworkHeader header(/*to node*/ other_node);
	bool ok = network.write(header,&payload,sizeof(payload));
	// Power down the radio. Note that the radio will get powered back up on the next write() call.
	radio.powerDown();
	digitalWrite(ActivePin, LOW);		// Turn off LED
}

