#!/usr/local/bin/python3
from PyQt5.QtCore import *
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *

import serial, glob, sys
from time import strptime, mktime
from datetime import datetime
import matplotlib.pyplot as plt
import numpy as np
from backend_qt5agg import FigureCanvasQTAgg as FigureCanvas
from matplotlib.figure import Figure

class MplCanvas(FigureCanvas):
    def __init__(self, parent=None, width=5, height=4, dpi=100):
        fig=Figure(figsize=(width, height), dpi=dpi)
        self.axes=fig.add_subplot(111)
        self.axes.hold(False)

        FigureCanvas.__init__(self, fig)
        self.setParent(parent)

        FigureCanvas.setSizePolicy(
            self,
            QSizePolicy.Expanding,
            QSizePolicy.Expanding
        )
        FigureCanvas.updateGeometry(self)

class abioSerial(QIODevice):
    BAUD_RATES=[50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800, 9600, 19200, 38400, 57600, 115200]

    _close=pyqtSignal()
    _write=pyqtSignal(str)

    class _Worker(QObject):
        dataAcquired=pyqtSignal(str)
        opened=pyqtSignal()
        closed=pyqtSignal()
        bytesWritten=pyqtSignal(int)

        def __init__(self, port, baud, parent=None, **kwargs):
            QObject.__init__(self, parent, **kwargs)

            self._serial=None
            self._timeout=0
            self._timerId=None

            self._port=port
            self._baud=baud

        @pyqtSlot()
        def open(self):
            self._serial=serial.Serial(self._port, self._baud, timeout=self._timeout)
            self._timerId=self.startTimer(1)
            self.opened.emit()

        @pyqtSlot()
        def close(self):
            self.killTimer(self._timerId)
            self._serial.close()
            self.closed.emit()

        @pyqtSlot(str)
        def write(self, data):
            _bytes=self._serial.write(bytes(data, "utf-8"))
            self.bytesWritten.emit(_bytes)

        def timerEvent(self, event):
            if self._timerId!=event.timerId(): return
            data=self._serial.read(1024)
            if not len(data): return
            self.dataAcquired.emit(data.decode("utf-8"))

    @staticmethod
    def ports():
        if sys.platform.startswith('win'):
            ports=['COM' + str(i + 1) for i in range(256)]
        elif sys.platform.startswith('linux') or sys.platform.startswith('cygwin'):
            ports=glob.glob('/dev/tty[A-Za-z]*')
        elif sys.platform.startswith('darwin'):
            ports=glob.glob('/dev/tty.*')
        else: raise EnvironmentError('Unsupported platform')

        result = []
        for port in ports:
            try:
                s=serial.Serial(port)
                s.close()
                result.append(port)
            except (OSError, serial.SerialException): pass
        return result

    def __init__(self, port, baud, parent=None, **kwargs):
        QIODevice.__init__(self, parent, **kwargs)

        self._open=False
        self._openMode=QIODevice.NotOpen
        self._thread=None
        self._worker=None
        self._data=""

        self._port=port
        self._baud=baud

    def openMode(self): return self._openMode

    def open(self, mode):
        if self._open or mode!=QIODevice.ReadWrite: return False
        self._openMode=mode

        self._thread=QThread(self, finished=self._threadFinished)
        self._worker=abioSerial._Worker(
            self._port,
            self._baud,
            opened=self._opened,
            closed=self._closed,
            dataAcquired=self._dataAcquired,
            bytesWritten=self._bytesWritten
        )

        self._thread.started.connect(self._worker.open)
        self._worker.closed.connect(self._thread.quit)
        self._close.connect(self._worker.close)
        self._write.connect(self._worker.write)

        self._worker.moveToThread(self._thread)
        self._thread.start()

    def isOpen(self): return self._open

    @pyqtSlot()
    def _opened(self): self._open=True

    @pyqtSlot()
    def close(self): self._close.emit()

    @pyqtSlot()
    def _closed(self): self._open=False

    @pyqtSlot()
    def _threadFinished(self):
        del self._worker
        self._worker=None
        del self._thread
        self._thread=None

    @pyqtSlot(str)
    def _dataAcquired(self, data):
        self._data+=data
        self.readyRead.emit()

    @pyqtSlot(int)
    def _bytesWritten(self, bytes): self.bytesWritten.emit(bytes)

    def read(self, maxSize=-1):
        if maxSize==-1: 
            data=self._data
            self._data=""
        else:
            data=self._data[:maxSize]
            self._data=self._data[maxSize:]
        return data

    def bytesAvailable(self): return len(self._data)>0

    def canReadLine(self): return '\n' in self._data

    @pyqtSlot(str)
    def write(self, data): self._write.emit(data) 

class abioLoggerWidget(QWidget):
    COLOR_NAMES=('r','g','b','c','m','y','k')

    def __init__(self, title, parent=None, **kwargs):
        QWidget.__init__(self, parent, **kwargs)

        self._title=title
        self._serial=None
        self._data=""
        self._plotData=[[]]
        self._numSensors=0
        self._labels=[]
        self._file=None

        l=QFormLayout(self)

        self._ports=QComboBox(self)
        for p in abioSerial.ports(): self._ports.addItem(p)
        l.addRow("Port:", self._ports)

        self._bauds=QComboBox(self)
        for b in abioSerial.BAUD_RATES: self._bauds.addItem(str(b))
        l.addRow("Baud Rate:", self._bauds)

        self._openButton=QPushButton("Open", self, clicked=self._openPort)
        l.addRow(self._openButton)

        self._log=QPlainTextEdit(self)
        l.addRow(self._log)

        self._output=QLineEdit(self)
        l.addRow(self._output)

        self._send=QPushButton("Send", self, clicked=self._sendData)
        l.addRow(self._send)

        self._canvas=MplCanvas(self, width=5, height=4, dpi=100)
        l.addRow(self._canvas)

    @pyqtSlot()
    def _openPort(self):
        if self._serial and self._serial.isOpen(): return
        self._serial=abioSerial(
            str(self._ports.currentText()),
            int(self._bauds.currentText()),
            self,
            readyRead=self._dataAvailable,
            bytesWritten=self._dataSent
        )
        self._serial.open(QIODevice.ReadWrite)

    @pyqtSlot()
    def _dataAvailable(self):
        data=self._serial.read()
        self._log.appendPlainText(data)

        self._data+=data
        if ";" not in self._data: return

        bits=self._data.split(';')
        self._data=""
        if not bits[-1].endswith(';'): self._data=bits.pop(-1)

        for bit in bits:
            args=bit.split(',')
            getattr(self, args[0])(*args[1:])

    def error(self, *args): print("ERROR!", args)

    def ready(self, numSensors):
        self._numSensors=int(numSensors)
        self._file=open("{0} {1}.csv".format(self._title, datetime.now()), "w")
        self._file.write("Date,Time,")
        for n in range(self._numSensors):
            self._plotData.append([])
            self._file.write("Temperature {0},".format(n+1))
            self._labels.append(QLabel(self))
            self.layout().addRow(
                "Temperature {0}:".format(n+1),
                self._labels[n]
            )
        self._file.write('\n')
        self._file.flush()

    def reading(self, date, time, *temps):        
            assert len(temps)==self._numSensors

            self._file.write("{0},{1},".format(date,time))

            dt=" ".join([date,time])
            self._plotData[0].append(datetime.fromtimestamp(mktime(strptime(dt, "%d/%m/%y %H:%M:%S"))))

            plotArgs=[]
            for n,t in enumerate(temps,1):
                self._plotData[n].append(float(t))
                plotArgs.extend([
                    self._plotData[0],
                    self._plotData[n],
                    abioLoggerWidget.COLOR_NAMES[n]
                ])
                self._labels[n-1].setText(u"{0}\u00B0C".format(t))
                self._file.write("{0},".format(t))
            self._file.write('\n')
            self._file.flush()

            self._canvas.axes.plot(*plotArgs)
            self._canvas.draw()

    @pyqtSlot()
    def _sendData(self):
        self._serial.write("{0}\n".format(self._output.text()))
        self._output.setText("")

    @pyqtSlot(int)
    def _dataSent(self, bytes):
        self._log.appendPlainText("{0} bytes sent".format(bytes))

    def closeEvent(self, event):
        if self._file:
            self._file.flush()
            self._file.close()
        if self._serial: self.serial.close()
        event.accept()

class ABIOLogger(QMainWindow):
    def __init__(self, title, parent=None, **kwargs):
        QMainWindow.__init__(self, parent, **kwargs)

        self.setWindowTitle("ABIO Data Logger: {0}".format(title))

        self._logger=abioLoggerWidget(title, self)
        self.setCentralWidget(self._logger)

if __name__=="__main__":
    from sys import argv, exit

    assert len(argv)==2

    a=QApplication(argv)
    abio=ABIOLogger(argv[1])
    abio.show()
    abio.raise_()
    exit(a.exec_())