package com.ardu.controllers;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Enumeration;

import com.ardu.tools.ActivePorts;
import com.sun.jna.Platform;

import gnu.io.CommPortIdentifier;
import gnu.io.SerialPort;
import gnu.io.SerialPortEvent;
import gnu.io.SerialPortEventListener;

/**
 * The Class ArduController.
 */
public class ArduController implements SerialPortEventListener {

    private static ActivePorts ap = new ActivePorts();

    /** The serial port. */
    public SerialPort serialPort;
    /** The port we're normally going to use. */
    private static final String PORT_NAMES[] = ap.getPortsAsArray();

    /** The input. */
    public static BufferedReader input;

    /** The output. */
    public static OutputStream output;

    /** Milliseconds to block while waiting for port open. */
    public static final int TIME_OUT = 2000;
    /** Default bits per second for COM port. */
    public static final int DATA_RATE = 9600;

    /**
     * Initialize.
     *
     * @throws Exception
     *             the exception
     */
    public void initialize() throws Exception {
        
        final File file = new File(ArduController.class.getProtectionDomain().getCodeSource().getLocation().getPath());
        
        System.setProperty("gnu.io.rxtx.SerialPorts", ap.getPortsAsString());
        

        if (Platform.isMac()) {
            addLibraryPath(file.getParent() + "/native/arduino/Mac_OS_X");
            System.out.println("Arduino mac");
        } else if (Platform.is64Bit()) {
            if (Platform.isLinux()) {
                addLibraryPath(file.getParent() + "/native/arduino/Linux/x86_64-unknown-linux-gnu");
                System.out.println("Arduino linux 64");
            } else if (Platform.isWindows()) {
                addLibraryPath(file.getParent() + "/native/arduino/Windows/64");
                System.out.println("Arduino windows 64");
            }
        } else if (Platform.isARM()) {
            if (Platform.isLinux()) {
                addLibraryPath(file.getParent() + "/native/arduino/Linux/arm");
                System.out.println("Arduino linux arm");
            } else if (Platform.isWindows()) {

                System.out.println("lib /arm/ftd2xx.dll");
            }
        } else {
            if (Platform.isLinux()) {
                addLibraryPath(file.getParent() + "/native/arduino/Linux/i686-unknown-linux-gnu");
                System.out.println("Arduino linux 32");
            } else if (Platform.isWindows()) {
                addLibraryPath(file.getParent() + "/native/arduino/Windows/i368-mingw32");
                System.out.println("lib /natives/i386/ftd2xx.dll");
            }
        }

        CommPortIdentifier portId = null;
        final Enumeration<?> portEnum = CommPortIdentifier.getPortIdentifiers();
        //System.out.println(portEnum.hasMoreElements());

        // First, Find an instance of serial port as set in PORT_NAMES.
        while (portEnum.hasMoreElements()) {
            final CommPortIdentifier currPortId = (CommPortIdentifier) portEnum.nextElement();

            for (final String portName : PORT_NAMES) {
                // System.out.println(portName);
                if (currPortId.getName().equals(portName)) {
                    portId = currPortId;
                    break;
                }
            }
        }

        if (portId == null) {
            System.out.println("Could not find COM port.");
            return;
        }

        try {
            // open serial port, and use class name for the appName.
            serialPort = (SerialPort) portId.open(this.getClass().getName(), TIME_OUT);

            // set port parameters
            serialPort.setSerialPortParams(DATA_RATE, SerialPort.DATABITS_8, SerialPort.STOPBITS_1,
                    SerialPort.PARITY_NONE);

            // open the streams
            input = new BufferedReader(new InputStreamReader(serialPort.getInputStream()));
            output = serialPort.getOutputStream();
            final char ch = 1;
            output.write(ch);

            // add event listeners
            serialPort.addEventListener(this);
            serialPort.notifyOnDataAvailable(true);
        } catch (final Exception e) {
            System.err.println(e.toString());
        }
    }

    /**
     * Close.
     */
    public synchronized void close() {
        if (serialPort != null) {
            serialPort.removeEventListener();
            serialPort.close();

        }
    }

    /*
     * (non-Javadoc)
     *
     * @see gnu.io.SerialPortEventListener#serialEvent(gnu.io.SerialPortEvent)
     */
    @Override
    public synchronized void serialEvent(final SerialPortEvent oEvent) {
        if (oEvent.getEventType() == SerialPortEvent.DATA_AVAILABLE) {
            try {
                final String inputLine = input.readLine();
                System.out.println(inputLine);

            } catch (final Exception e) {
                System.err.println(e.toString());
            }
        }

    }

    /**
     * Write data.
     *
     * @param data
     *            the data
     */
    public static synchronized void writeData(final String data) {
        System.out.println("Sent: " + data);
        try {
            output.write(data.getBytes());
            output.flush();
        } catch (final Exception e) {
            System.out.println("could not write to port");
        }
    }

    public static void sendShortString() throws Exception {

        final ArduController main = new ArduController();
        main.initialize();
        final Thread t = new Thread() {
            @Override
            public void run() {
                // the following line will keep this app alive for 1000 seconds,
                // waiting for events to occur and responding to them (printing
                // incoming messages to console).
                BufferedReader br = null;
                br = new BufferedReader(new InputStreamReader(System.in));
                while (true) {
                    try {
                        // Thread.sleep(1000);

                        writeData(br.readLine());

                    } catch (final IOException ie) {
                        ie.printStackTrace();
                    }
                }

            }
        };
        t.start();
        System.out.println("Started");

    }

    /**
     * The main method.
     *
     * @param args
     *            the arguments
     * @throws Exception
     *             the exception
     */
    public static void main(final String[] args) throws Exception {

        sendShortString();

    }

    /**
     * Adds the library path.
     *
     * @param pathToAdd
     *            the path to add
     * @throws Exception
     *             the exception
     */
    public static void addLibraryPath(final String pathToAdd) throws Exception {
        final Field usrPathsField = ClassLoader.class.getDeclaredField("usr_paths");
        usrPathsField.setAccessible(true);

        final String[] paths = (String[]) usrPathsField.get(null);

        for (final String path : paths) {
            if (path.equals(pathToAdd)) {
                return;
            }
        }

        final String[] newPaths = Arrays.copyOf(paths, paths.length + 1);
        newPaths[newPaths.length - 1] = pathToAdd;
        usrPathsField.set(null, newPaths);
    }

}
