//*****************************************
// Microdot V1
// Copyright 2006 Philip Pulle
//                RGB Sunset Productions
// For private use only, not for commercial
// use. 
// ****************************************

#include <system.h>
#include "display.h"
#include "control.h"
#include "t1rtc.h"
#include "adc.h"
#include "rand.h"


extern unsigned char nRTCSecs;
extern unsigned char nRTCMins;
extern unsigned char nRTCHours;
extern unsigned char nRTCTick;
unsigned int nBrightness;

//                              LED1  LED2  LED3  LED4  LED5  LED6  LED7  LED8  LED9  LED10 LED11 LED12 LED13 LED14 LED15 LED16 LED17 LED18 LED19 LED20 LED21 LED22 LED23 LED24 LED25 LED26 LED27 LED28 LED29 LED30
unsigned char LEDS_PORTA[31] = { 0x10, 0x00, 0x00, 0x00, 0x04, 0x02, 0x00, 0x10, 0x08, 0x02, 0x10, 0x00, 0x02, 0x02, 0x08, 0x04, 0x02, 0x10, 0x04, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x04, 0x10, 0x08, 0x04, 0x08, 0x00};
unsigned char LEDS_TRISA[31] = { 0xef, 0xff, 0xff, 0xef, 0xeb, 0xf5, 0xfd, 0xeb, 0xf5, 0xfd, 0xed, 0xf7, 0xf9, 0xed, 0xf7, 0xf9, 0xfd, 0xe7, 0xfb, 0xfd, 0xe7, 0xfb, 0xfb, 0xef, 0xf7, 0xfb, 0xef, 0xf7, 0xf3, 0xf3, 0xff};
unsigned char LEDS_PORTB[31] = { 0x00, 0x02, 0x04, 0x02, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x04, 0x02, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
unsigned char LEDS_TRISB[31] = { 0xfd, 0xf9, 0xf9, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xfb, 0xfb, 0xff, 0xfb, 0xfd, 0xfb, 0xfb, 0xfd, 0xfb, 0xfb, 0xff, 0xff, 0xff};
unsigned char nLedsA[30];
unsigned char nLedsB[30];
unsigned char nCurrentLed=0;
unsigned char nCurrentBuffer=0;
unsigned char nCurrentZBuffer=1;
unsigned char nShowBuf = 0;

//Variables showing the number of LEDs to light in each colour group
unsigned char nG4,nG3,nG2,nG1;
//Variables defining position in transition
unsigned char nDuty, nCycle;

//display buffers
unsigned char *leds_porta;
unsigned char *leds_portb;
unsigned char *leds_trisa;
unsigned char *leds_trisb;
unsigned char *leds_portend;
unsigned char *leds_trisend;
unsigned char *ledsa;
unsigned char *ledsb;

//FUNCTION to set the brightness of the display
//INPUTS:
//	Brightness :=0 full brightness
//	Brightness<227 :less bright, higher = dimmer
void set_brightness(unsigned int Brightness)
{
		nBrightness = Brightness + NUM_LEDS-1;
}

//FUNCTION to initialise the display
void init_display(void)
{
	unsigned char i;
	
	//Turn all LEDs off
	for (i=0;i<NUM_LEDS;i++)
	{
		nLedsA[i] = LED_OFF;
		nLedsB[i] = LED_OFF;
	}
	nCycle = 0;
	nDuty = 0;
	leds_porta = LEDS_PORTA;
	leds_portb = LEDS_PORTB;
	leds_trisa = LEDS_TRISA;
	leds_trisb = LEDS_TRISB;
	leds_trisend = LEDS_TRISA+NUM_LEDS;
	leds_portend = LEDS_PORTA+NUM_LEDS;
	ledsa = nLedsA;
	ledsb = nLedsB;
	
	//Setup the ADC to read the ambient light sensor
	adcon1 = 0b00000000;	//left justify result
	adcon0 = 0b00000001;
	set_bit(ansel,0);
	set_brightness(0);
	//Setup TMR0 to update the display
	
	option_reg = 0b10001000;	//RBPU disbled, T0 internal
	set_bit(intcon, TMR0IE);	//enable the display interrupt
	set_bit(intcon, GIE);		//enable all interrupts
	
}


//FUNCTION to turn on each led consequtively
void led_test(void)
{
	unsigned char i;
	for(i=0;i<30;i++)
	{
		trisa = 0xff;
		trisb = 0xff;
		porta = LEDS_PORTA[i];
		portb = LEDS_PORTB[i];
		trisa = LEDS_TRISA[i];
		trisb = LEDS_TRISB[i];
		delay_ms(20);
	}
}


//FUNCTION to get the current led on/off status
//INPUTS:
//	nLed :- number of LED to query
//where return:= {LED_ON, LED_OFF}
unsigned char get_led(unsigned char nLed)
{
	if(nCurrentBuffer==0)
		return nLedsA[nLed];
	else
		return nLedsB[nLed];
}

//FUNCTION to turn on a LED on the display
//INPUTS: 
//	nLed :- number of LED to turn on
void led_on(unsigned char nLed)
{
	if(nCurrentBuffer==0)
		nLedsA[nLed] = LED_ON;
	else
		nLedsB[nLed] = LED_ON;
}

//FUNCTION to turn on a LED on the display
//INPUTS: 
//	nLed :- number of LED to turn on
void led_off(unsigned char nLed)
{
	if (nCurrentBuffer==0)
		nLedsA[nLed] = LED_OFF;
	else
		nLedsB[nLed] = LED_OFF;
}

//FUNCTION to perform a transition between display buffers
//and switch between them at finish
void set_transition(void)
{
	unsigned char nBuf,i;
						

	nCycle = 0;
	for(i=0;i<MAX_DUTY;i++)
	{
		nDuty = i;
		delay_ms(5);
	}
	nDuty = 0;
		nBuf = nCurrentZBuffer;
	nCurrentZBuffer = nCurrentBuffer;
	nCurrentBuffer = nBuf;

}

//FUNCTION to set transition flag to max cycle...ie full brightness
void set_notransition(void)
{
	nDuty = MAX_DUTY;
}

//FUNCTION to select the next LED in the list and either turn it on
//or leave it off according to the LED table. Should be called periodically
//by the interrupt function
void update_display(void)
{
	unsigned char i;
	unsigned char nBuf;
	unsigned char *ptr;
	//Move to next LED
	nCurrentLed++;
	ledsa++;
	ledsb++;
	leds_porta++;
	leds_portb++;
	leds_trisa++;
	leds_trisb++;
	i = ALL_OFF;
	
	nBuf = nShowBuf;
	if(nCurrentLed>(NUM_LEDS-1)) 
	{
		
		if (nCurrentLed>nBrightness)
		{
			nCurrentLed = 0;	//rollover if necessary
			leds_porta = LEDS_PORTA;
			leds_portb = LEDS_PORTB;
			leds_trisa = LEDS_TRISA;
			leds_trisb = LEDS_TRISB;
			ledsa = nLedsA;
			ledsb = nLedsB;
			nCycle++;
			nCycle = nCycle & (MAX_DUTY-1);

			if(nDuty>nCycle)
				nShowBuf = nCurrentBuffer;
			else 	
				nShowBuf = nCurrentZBuffer;
				
			if(nShowBuf==0)
			{	
				if(*ledsa==LED_ON)
					i = nCurrentLed;
			}
			else
			{
				if(*ledsb==LED_ON)
					i=nCurrentLed;
			}
		}
	}
	else
	{
		if(nBuf==0)
		{	
			if(*ledsa==LED_ON)
				i = nCurrentLed;
		}
		else
		{
			if(*ledsb==LED_ON)
				i=nCurrentLed;
		}
	}
	
	
	porta = 0;		//turn off previous leds what ever they may be
	portb = 0;
	
	if(i==ALL_OFF)
	{
		trisa = *leds_trisend;
		trisb = *leds_trisend;
		porta = *leds_portend;
		portb = *leds_portend;
	}
	else
	{
		trisa = *leds_trisa;
		trisb = *leds_trisb;
		porta = *leds_porta;
		portb = *leds_portb;
	}

}

//FUNCTION to calculate time from RTC and update display variables
void calc_time(void)
{
	unsigned char i,j;

	i = nRTCHours; j = nRTCMins;
	nG4 = i / 10 ;
	nG3 = i % 10 ;
	
	nG2 = j / 10 ;
	nG1 = j % 10 ;
}



//FUNCTION to show the time
//INPUTS:
//	nG4..nG1 	:- number of LEDs in colour groups (red, amber, green, blue)
//	nMode 		:- DISP_RANDOM, DISP_LINEAR show either random distribution (for time) of 
//				Leds or a linear distribution (for counting, stopwatch)
void show_time(unsigned char nMode)
{
	unsigned char i,j,k;
	//bound parameters to make sure bogus values don't send algorithm into la la land
	if (nG4>5) nG4 = 5;
	if (nG3>9) nG3 = 9;
	if (nG2>5) nG2 = 5;
	if (nG1>9) nG1 = 9;	
	if (nMode==DISP_LINEAR)
	{
		//Do red
		for (i = 0; i<6; i++)
		{
			if (i<nG4) led_on(i); else led_off(i);
		}
		//Do Amber
		j = nG3 + 6;
		for(i =6; i<15; i++)
		{
			
			if (i<j) led_on(i); else led_off(i);
		}
		//Do green
		j = nG2 + 15;
		for(i=15; i<21; i++)
		{
			
			if(i<j) led_on(i); else led_off(i);
		}
		//Do Blue
		j = nG1 + 21;
		for(i=21;i<30;i++)
		{
			if(i<j) led_on(i); else led_off(i);
		}
	}
	else
	{
		//Red
		for(i=0;i<6;i++) led_off(i);
		j = nG4;
		for(i=0;i<j;i++)
		{
			k = rand() % 6;			//pick a random position
			if(get_led(k)==LED_ON)	//already set
				i--;				//so try again till we've cleared j items
			else
				led_on(k);	//set this item
		}
		
		//amber
		for(i=6;i<15;i++) led_off(i);
		j = nG3+6;
		for(i=6;i<j;i++)
		{
			k = rand() % 9;			//pick a random position
			k = k + 6;
			if(get_led(k)==LED_ON)	//already set
				i--;				//so try again till we've set j items
			else
				led_on(k);		//set this item
		}
		
		//green
		for(i=15;i<21;i++) led_off(i);
		j = nG2+15;
		for(i=15;i<j;i++)
		{
			k = rand() % 6;			//pick a random position
			k = k + 15;
			if(get_led(k)==LED_ON)	//already set
				i--;				//so try again till we've set j items
			else
				led_on(k);	//clear this item
		}
		
		//mins
		for(i=21;i<30;i++) led_off(i);
		j = nG1+21;
		for(i=21;i<j;i++)
		{
			k = rand() % 9;			//pick a random position
			k = k + 21;
			if(get_led(k)==LED_ON)	//already set
				i--;				//so try again till we've set j items
			else
				led_on(k);	//set this item
		}	
				
	}

}

//FUNCTION to turn on all LEDs briefly
void flash_display(void)
{
	unsigned char i;
	for(i=0;i<30;i++) led_on(i);
	delay_ms(40);
	for(i=0;i<30;i++) led_off(i);
}

//FUNCTION to flash all LEDs in a colour group briefly
//INPUTS:
//	nGroup :- group to flash {0..3} 0=red, 1=yellow, 2=green, 3=blue
void flash_group(unsigned char nGroup)
{
	unsigned char i,j,k;
	switch(nGroup)
	{
		case 0:
			j = 0; k = 6;
			break;
		case 1:
			j = 6; k = 15;
			break;
		case 2:
			j = 15; k = 21;
			break;
		case 3:
			j = 21; k = 30;
			break;

	}
	for(i=j;i<k;i++) led_on(i);
	delay_ms(20);
	for(i=j;i<k;i++) led_off(i);	
}

//FUNCTION to check and adjust the brightness of the display
void adjust_brightness(void)
{
	unsigned short nAmbient;
	unsigned char i;
	
	//Dont adjust function if nearby LEDs are on
	if (nLedsA[3]==LED_ON || nLedsB[3]==LED_ON) return;
	
	nAmbient = adc_measure(0);
	HIBYTE(i,nAmbient);
	i = 256 - i;
	if (i<AMB_MIN) i = 0;
	i = i / 2;
	set_brightness(i);
	
}