#include <EEPROM.h>

// Sketch for bike light patterns and 12V step-up converter's MOSFET driver.
// The ATTiny remembers the previous brightness level set with the momentary switch.
// The LED strips blink at 17 Hz for night time mode visibility.

#define MAX_BOOST_DUTY 0.80 // Maximum duty cycle for the boost converter

// Set the boost converter's minimum and maximum PWM levels
const int minBoostOutput = 0; 
const int maxBoostOutput = 255 * MAX_BOOST_DUTY;
int currentBoostOutput; // Remembers the PWM level for the boost converter

// Pins for the ATTiny85
const int ledOutputPin=1; // DIP pin 6
const int boostOutputPin=0; // DIP pin 5
const int buttonPin=3; // DIP pin 2
const int voltageFbPin=A1; // DIP pin 7

const int maxBrightness=250;
const int brightnessAddress=0;
const int brightnessSteps=5;
int minBrightness=maxBrightness/brightnessSteps;
int brightnessStepSize=maxBrightness/brightnessSteps;

// Recalls the brightness settings for each color
int brightness=constrain(EEPROM.read(brightnessAddress),minBrightness,maxBrightness);

int togglePeriod=30; // Sets the flicker frequency. togglePeriod = 1/(2f)
unsigned long lastToggleTime; // Remembers when output last toggled
boolean ledState; // Remembers if LED was on or off

// Variables for button debouncing
boolean buttonReading;
int buttonState;             // the current reading from the input pin
int lastButtonState = LOW;   // the previous reading from the input pin
long lastDebounceTime = 0;
long debounceDelay = 50; 


void setup() {

  TCCR0A = _BV(COM0A1)|_BV(WGM01)|_BV(WGM00);
  TCCR0B = _BV(CS00);
  currentBoostOutput=minBoostOutput;
  OCR0A = minBoostOutput; // Soft starts converter

  analogReference(INTERNAL); // Uses internal 1.1V reference

  pinMode(buttonPin,INPUT_PULLUP);
  pinMode(voltageFbPin,INPUT);
  pinMode(ledOutputPin,OUTPUT);
  pinMode(boostOutputPin,OUTPUT);
  
  analogWrite(ledOutputPin,brightness);
}


void loop() {

  // Flickers the LEDs
  if(millis()-lastToggleTime>togglePeriod){
    lastToggleTime=millis();

    if(ledState==LOW){
      pinMode(ledOutputPin,OUTPUT);
      ledState=HIGH; 
    }
    else if (ledState==HIGH){
      OCR0A = minBoostOutput; // Turn off step-up converter before turning LEDs off for overvoltage protection.
      pinMode(ledOutputPin,INPUT);
      ledState=LOW;
    }
  }

  if (ledState==HIGH){ // Turn step-up converter back on for the LEDs
    OCR0A=currentBoostOutput;
    boostConverter();
  }

  buttonReading=digitalRead(buttonPin);
  
// Debouncing code
  if (buttonReading != lastButtonState) {
    lastDebounceTime = millis();
  }

  if ((millis() - lastDebounceTime) > debounceDelay) {

    if (buttonReading != buttonState) {
      buttonState = buttonReading;

      if (buttonState == HIGH) {
        // If button is pressed and released, increment counter until it overflows
        if(brightness<maxBrightness){
          brightness+=brightnessStepSize;
          brightness=constrain(brightness,minBrightness,maxBrightness);
        }
        else if(brightness==maxBrightness){
          brightness=minBrightness; 
        }

        analogWrite(ledOutputPin,brightness); // Set brightness to new value
        EEPROM.write(brightnessAddress,brightness); // Remember brightness setting
      }
    }
  }
  lastButtonState = buttonReading;
}

