/*
This code controls the output of a pin (which in turn controls a length of EL wire) 
based on the readings from an accelerometer (Memsic 2125). The accelerometer is used
to roughly determine whether the person wearing it is running, walking, or standing still. 
The code determines the maximum and minimum readings over a one second period to get
the "range" of the person's motions (more "violent" movement like running produces a 
larger range then walking, which is more violent than standing still). From this range,
the Arduino determines what activity is occurring and changes the status of the 
EL wire pin accordingly. A serial connection is setup for debugging the code.

created 03/18/14
*/

//Initialize Variables
const int xpin = 7;                   // x-axis 
const int ypin = 6;                   // y-axis
const int ELwire = 13;                //pin controlling EL wire
int counter = 0;                      //set initial counter value
int activity = 0;                     //set initial activity
int length = 6;                       //set number of samples

int x_raw;                            //define variables for later
int y_raw;
int x_conv;
int y_conv;
int max_val;
int min_val;
int range;

int X[6];                            //initialize arrays for storing accel readings
int Y[6];


void setup(){                         
  //Serial.begin(9600);                 //setup serial connection for debugging
  pinMode(ELwire, OUTPUT);            //initialize pins
  pinMode(xpin, INPUT);
  pinMode(ypin, INPUT);
  digitalWrite(ELwire, LOW);          //start with EL wire off
}


void loop(){ 
  x_raw = pulseIn(xpin, HIGH);        //reads incoming pulse, gives length in microseconds
  y_raw = pulseIn(ypin, HIGH);
  
  x_conv = ((0.001*x_raw*0.1) - 0.5)*8*100;  //convert from microsec to grav (see Memsic datasheet)
  y_conv = ((0.001*y_raw*0.1) - 0.5)*8*100;
  
  X[counter] = x_conv;                   //assign new value to space in array
  Y[counter] = y_conv;
  
  if (counter==length){                  //after some time (~1 sec), update activity
    for (int i=0; i<=length; i++){       //for all items in array, check max/min
      if (i == 0){                       //for first item only
        max_val = max(X[i], X[i+1]);
        min_val = min(X[i], X[i+1]);
      }
      else {                             //for all other items
        max_val = max(max_val, X[i]);    //determine new max
        min_val = min(min_val, X[i]);    //determine new min
      }
    }
    range = max_val - min_val;           //find range
    
    if (range>=30){                      //running (30+)
        activity = 1;
      }
    else if (range>=15 && range<=30){    //walking (15-30)
        activity = 2;
      }
    else{                                //little/no movement
        activity = 0;
      }
    counter = 0;                         //reset counter after activity is determined
    }
  
  else {                                 //if counter<10, update counter and get more values for arrays
    counter++; 
  }
  
  switch(activity){                 //check activity and set EL wire to correct activity
  case 0:                           //little/no movement -- off
    digitalWrite(ELwire, LOW);      //off
    break;
  case 1:                           //run -- steady on
    digitalWrite(ELwire, HIGH);     //on
    break;
  case 2:                           //walk -- flashes
    if ((counter%2) == 0){          //if modulus is 0 (even value)
      digitalWrite(ELwire, HIGH);   //turn on
    }
    else {                          //if odd
      digitalWrite(ELwire, LOW);    //turn off
    }
    break;
  }
  
  /*Serial.print(counter);         //serial printouts for debugging
  Serial.print("\t");
  Serial.print(max_val);
  Serial.print("\t");
  Serial.print(min_val);
  Serial.print("\t");
  Serial.print(range);
  Serial.print("\t");
  Serial.print(activity);
  Serial.println();*/
  
  delay(100);                      //delay for 100 ms 
}
