/*
 * Code for bike light with 3A custom made LED driver and two 15mA LED driver.
 * Before uploading, select ATTiny84, 20 MHz (external 4.3V BOD) under tools. BOD is needed for EEPROM to be used. If you don't want to use EEPROM, disable it.
 * 
 * Boost mode button temporarily sets the white LEDs to maximum brightness for 3 minutes.
 * 
 * To turn on lamp, hold power button for medium duration.
 * To turn off lamp, hold power button for medium duration.
 * To select battery chemistry, hold power button for long duration.
 * To increment mode, hold power button for short duration.
 * To select mode type, hold power button for long duration.
 * To turn on boost mode or extend it, hold the boost mode button for a short duration.
 * To turn off boost mode, hold the boost mode button for medium duration.
 * 
 * Battery monitor set up for 3 Li-ion and 3 LiFePO4 battery configurations.
 * Drive currents calculated for 0.22 ohm Isense resistor. If a different value is used, recalculate their values. Aref of 1.1V used and ADC ranges between 0 and 1023.
 * ADC=Idrive*Rsense*1023/1.1
 */ 

#include <EEPROM.h>
#include <avr/sleep.h>

// Utility macros
#define adc_disable() (ADCSRA &= ~(1<<ADEN)) // disable ADC
#define adc_enable()  (ADCSRA |=  (1<<ADEN)) // re-enable ADC

// Pins
byte powerButtonPin = 8;
byte sidesPwmPin = 7;
byte driverFbPin = A6;
byte driverPwmPin = 5;
byte boostButtonPin = 4;
byte batteryMonitorPin = A0;

// Buttons
boolean powerButtonState;
boolean boostButtonState;
const int debounceDelay = 255;
byte delayCounts = 0;

// Headlight using buck LED driver.
int onTime = 0;
const int period = 300; // Determines the frequency and resolution.
const int maxOnTime = period; // Keep it equal to period for buck driver. Can decrease for current limiting
int feedbackReading;
int desiredFeedback; // Depends on Rsense and drive current.
unsigned long boostDuration = 180000; // Boost mode duration in milliseconds. 3 minutes.
//unsigned long boostDuration = 20000; // Test for 20 seconds
boolean boostExtendAllowed;
boolean boostMode;
int frontBrightnessHigh, frontBrightnessLow, sidesBrightness;

// Bike light patterns.
byte mode = 5; // Start with low battery mode.
byte modeType = 1;
unsigned long elapsedMillis0, elapsedMillis1, elapsedMillis2;
unsigned long prevMillis0, prevMillis1, prevMillis2;
const int stepTime = 10; // Actual time in milliseconds is stepTime/68. Fade frequency is approximately 2.5 Hz.
byte stepLevel;
const int stepTop = 39;
// For side light partial fade.
const byte sideFadeLevels[40] = {31, 32, 33, 36, 40, 45, 51, 58, 67, 76, 87, 99, 112, 126, 141, 157, 174, 193, 212, 233, 255, 233, 212, 193, 174, 157, 141, 126, 112, 99, 87, 76, 67, 58, 51, 45, 40, 36, 33, 32};
/*
  ADC values for different battery chemistries.
  AREF=1.1V, resistors of 1M and 56k.
   3 cell LiFePO4: Low=8.4V (2.8V/cell); Dead:8.1V(2.7V/cell); Readings:0.445V,0.430V; ADC:414,399
   3 cell Li-ion: Low=10.2V (3.4V/cell); Dead:9.9V (3.3V/cell); Readings:0.541V,0.525V; ADC:503,488

   Second option. Closer spacing.
   3 cell LiFePO4: Low=8.4V (2.8V/cell); Dead:8.3V(2.77V/cell); Readings:0.445V,0.440V; ADC:414,409
   3 cell Li-ion: Low=10.2V (3.4V/cell); Dead:10.1V (3.3V/cell); Readings:0.541V,0.535V; ADC:503,498
*/
// Battery monitor for 3s Li-ion or 3s LiFePO4. 1.1V reference used.
const int lifepo4LowADC = 414;
const int lifepo4deadADC = 409;
const int liionLowADC = 503;
const int liionDeadADC = 498;
int lowBattADC;
int deadBattADC;
boolean liionSelected = EEPROM.read(0); // true for 3.7V Li-ion batteries, false for 3.2V LiFePO4 batteries
int batteryReading;

void setup() {
  pinMode(powerButtonPin, INPUT_PULLUP);
  pinMode(batteryMonitorPin, INPUT);
  pinMode(driverFbPin, INPUT);
  pinMode(driverPwmPin, OUTPUT);
  analogReference(INTERNAL);
  sleepNow();
}

void loop() {
  readButtons();
  boostModeTimer();

  switch (mode) {
    // Default modes.
    case 0: // Fade sides only.
      OCR1B = 0; // To prevent pin from floating. Front light should only dimly glow.
      sidesFade();
      batteryReading = analogRead(batteryMonitorPin);
      break;
    // Fade sides for modes 1 through 4
    case 1: // High low level 0
      if (boostMode == true) {
        frontBrightnessHigh = 590; // For Rsense value of 0.235 ohm and Idrive of 2.7A. Vfb of 634mV used. Use ADC of 590
        frontBrightnessLow = 5; // 23mA
        fiftyFifty();
      }
      else {
        frontBrightnessHigh = 40; // 180mA
        frontBrightnessLow = 4; // 18mA
        fiftyFifty();
      }
      LEDDriver();
      sidesFade();
      break;
    case 2: // Inverted triple pulse level 1.
      if (boostMode == true) {
        frontBrightnessHigh = 590; // 2.7A
        frontBrightnessLow = 250; // 1.1A
        invertedTriplePulse();
      }
      else {
        frontBrightnessHigh = 70; // 320mA
        frontBrightnessLow = 35; // 160mA
        invertedTriplePulse();
      }
      LEDDriver();
      sidesFade();
      break;
    case 3: // Inverted triple pulse level 2.
      if (boostMode == true) {
        frontBrightnessHigh = 590; // 2.7A
        frontBrightnessLow = 250; // 1.1A
        invertedTriplePulse();
      }
      else {
        frontBrightnessHigh = 140; // 640mA
        frontBrightnessLow = 70; // 320mA
        invertedTriplePulse();
      }
      LEDDriver();
      sidesFade();
      break;
    case 4: // Inverted triple pulse level 3
      if (boostMode == true) {
        frontBrightnessHigh = 590; // 2.7A
        frontBrightnessLow = 250; // 1.1A
        invertedTriplePulse();
      }
      else {
        frontBrightnessHigh = 280; // 1.3A
        frontBrightnessLow = 140; // 640A
        invertedTriplePulse();
      }
      LEDDriver();
      sidesFade();
      break;
    // Other modes.
    case 5: // Low battery. Solid sides only
      OCR1B = 0;
      analogWrite(sidesPwmPin, 63);
      batteryReading = analogRead(batteryMonitorPin);
      break;
    // Turn off sides for modes 6 through 9
    case 6: // Solid front level 0, no sides
      desiredFeedback = 5; // 23mA
      analogWrite(sidesPwmPin, 0);
      LEDDriver();
      break;
    case 7: // Solid front level 1, no sides
      desiredFeedback = 25; // 110mA
      analogWrite(sidesPwmPin, 0);
      LEDDriver();
      break;
    case 8: // Solid front level 2, no sides
      desiredFeedback = 150; // 690mA
      analogWrite(sidesPwmPin, 0);
      LEDDriver();
      break;
  }

  if (batteryReading < lowBattADC) { // If battery is low, go to low battery mode.
    mode = 5;
    modeType = 1;
  }
  if (batteryReading < deadBattADC) { // If battery is considered dead, go to sleep.
    sleepNow();
  }
}

