
//Including Libs
#include <OneWire.h>
#include <dht11.h>
#include <VirtualWire.h>
#include <Wire.h> //Pressure sensor

//Defining Pins
dht11 DHT11;

#define DHT11PIN 2
#define ONE_WIRE_BUS 10

#define BMP085_ADDRESS 0x77  // I2C address of BMP085 //Pressure sensor

#define INTEGER_MAX (pow(2,31)-1)
#define E_MAX (pow(10, 7))
#define E_MIN (pow(10, -6))
#define EPSILON 0.000000119209

OneWire oneWire(ONE_WIRE_BUS);

int RF_TX_PIN = 12;

//Pressure sensor
const unsigned char OSS = 0;  // Oversampling Setting

// Calibration values
int ac1;
int ac2;
int ac3;
unsigned int ac4;
unsigned int ac5;
unsigned int ac6;
int b1;
int b2;
int mb;
int mc;
int md;

// b5 is calculated in bmp085GetTemperature(...), this variable is also used in bmp085GetPressure(...)
// so ...Temperature(...) must be called before ...Pressure(...).
long b5; 

//Seting up
void setup()
{
  //For Debug
  Serial.begin(9600);
  Serial.println("Setup");

  //Pressure sensor
  Wire.begin();
  bmp085Calibration();

  // Setup transmit pin
  vw_set_tx_pin(RF_TX_PIN); 
  vw_setup(2000); //

  //For Debug
  Serial.println("Start Pressure");
}

//Main Loop
void loop()
{
  Serial.println("\n");

  //First Sensor
  //Humidity
  int chk = DHT11.read(DHT11PIN);

  //Send Data #H is for Humidity
  SendData("#H"+ ((String)DHT11.humidity));



  //Seccond Sensor
  //Temp (Baro Pressure)
  float temperature = bmp085GetTemperature(bmp085ReadUT()); //MUST be called first 

  //Send Data #C is for Celcious
  SendData("#C"+(String(temperature,2)));



  //Same seccond Sensor, next output
  //Baro Pressure
  float pressure = bmp085GetPressure(bmp085ReadUP());

  //Send Data #P is for pressure
  SendData("#P"+(String(pressure/100,2)));


  //Wait for Next Loop
  delay(5000);
}

void SendData(String Data)
{
  //Debug
  Serial.println("-->"+ Data + "<-- ");

  //Making char Array of String
  const char* rawdata = Data.c_str();

  digitalWrite(13, true); // Flash a light to show transmitting
  vw_send((uint8_t *)rawdata, strlen(rawdata)); //Send Data
  vw_wait_tx(); // Wait until the whole message is gone
  digitalWrite(13, false);
}

//Calabration of Barometic sensor
void bmp085Calibration()
{
  ac1 = bmp085ReadInt(0xAA);
  ac2 = bmp085ReadInt(0xAC);
  ac3 = bmp085ReadInt(0xAE);
  ac4 = bmp085ReadInt(0xB0);
  ac5 = bmp085ReadInt(0xB2);
  ac6 = bmp085ReadInt(0xB4);
  b1 = bmp085ReadInt(0xB6);
  b2 = bmp085ReadInt(0xB8);
  mb = bmp085ReadInt(0xBA);
  mc = bmp085ReadInt(0xBC);
  md = bmp085ReadInt(0xBE);
}

// Calculate temperature in deg C
float bmp085GetTemperature(unsigned int ut){
  long x1, x2;

  x1 = (((long)ut - (long)ac6)*(long)ac5) >> 15;
  x2 = ((long)mc << 11)/(x1 + md);
  b5 = x1 + x2;

  float temp = ((b5 + 8)>>4);
  temp = temp /10;

  return temp;
}

// Calculate pressure given up
// calibration values must be known
// b5 is also required so bmp085GetTemperature(...) must be called first.
// Value returned will be pressure in units of Pa.
long bmp085GetPressure(unsigned long up){
  long x1, x2, x3, b3, b6, p;
  unsigned long b4, b7;

  b6 = b5 - 4000;
  // Calculate B3
  x1 = (b2 * (b6 * b6)>>12)>>11;
  x2 = (ac2 * b6)>>11;
  x3 = x1 + x2;
  b3 = (((((long)ac1)*4 + x3)<<OSS) + 2)>>2;

  // Calculate B4
  x1 = (ac3 * b6)>>13;
  x2 = (b1 * ((b6 * b6)>>12))>>16;
  x3 = ((x1 + x2) + 2)>>2;
  b4 = (ac4 * (unsigned long)(x3 + 32768))>>15;

  b7 = ((unsigned long)(up - b3) * (50000>>OSS));
  if (b7 < 0x80000000)
    p = (b7<<1)/b4;
  else
    p = (b7/b4)<<1;

  x1 = (p>>8) * (p>>8);
  x1 = (x1 * 3038)>>16;
  x2 = (-7357 * p)>>16;
  p += (x1 + x2 + 3791)>>4;

  long temp = p;
  return temp;
}

// Read 1 byte from the BMP085 at 'address'
char bmp085Read(unsigned char address)
{
  unsigned char data;

  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(address);
  Wire.endTransmission();

  Wire.requestFrom(BMP085_ADDRESS, 1);
  while(!Wire.available())
    ;

  return Wire.read();
}

// Read 2 bytes from the BMP085
// First byte will be from 'address'
// Second byte will be from 'address'+1
int bmp085ReadInt(unsigned char address)
{
  unsigned char msb, lsb;

  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(address);
  Wire.endTransmission();

  Wire.requestFrom(BMP085_ADDRESS, 2);
  while(Wire.available()<2)
    ;
  msb = Wire.read();
  lsb = Wire.read();

  return (int) msb<<8 | lsb;
}

// Read the uncompensated temperature value
unsigned int bmp085ReadUT(){
  unsigned int ut;

  // Write 0x2E into Register 0xF4
  // This requests a temperature reading
  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(0xF4);
  Wire.write(0x2E);
  Wire.endTransmission();

  // Wait at least 4.5ms
  delay(5);

  // Read two bytes from registers 0xF6 and 0xF7
  ut = bmp085ReadInt(0xF6);
  return ut;
}

// Read the uncompensated pressure value
unsigned long bmp085ReadUP(){

  unsigned char msb, lsb, xlsb;
  unsigned long up = 0;

  // Write 0x34+(OSS<<6) into register 0xF4
  // Request a pressure reading w/ oversampling setting
  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(0xF4);
  Wire.write(0x34 + (OSS<<6));
  Wire.endTransmission();

  // Wait for conversion, delay time dependent on OSS
  delay(2 + (3<<OSS));

  // Read register 0xF6 (MSB), 0xF7 (LSB), and 0xF8 (XLSB)
  msb = bmp085Read(0xF6);
  lsb = bmp085Read(0xF7);
  xlsb = bmp085Read(0xF8);

  up = (((unsigned long) msb << 16) | ((unsigned long) lsb << 8) | (unsigned long) xlsb) >> (8-OSS);

  return up;
}













